<?php
    // ==========================================
    // Clase: InformeCierreCaja
    // Autor: John Christian Tello Vilca
    // Descripción: Representa el cierre diario de caja.
    // Tabla: INFORME_CIERRE_CAJA
    // Nota: Los campos REPORT_TARJ_MENOS_PROP_TARJ, EFECTIVO_1 y
    //       EFECTIVO_SISTEMA_FINAL son calculados en SQL Server.
    //       Aquí incluimos helpers para recalcularlos en memoria.
    // ==========================================
    class InformeCierreCaja
    {
        // ======= Atributos (columnas) =======
        private $idCierreCaja;                 // INT IDENTITY(1,1) PK
        private $fechaRegCaja;                 // DATETIME DEFAULT GETDATE()

        private $totalReportPos;               // DECIMAL(10,2)
        private $totalPropTarjeta;             // DECIMAL(10,2)
        private $totalPropTarj095;             // DECIMAL(10,2)  -- 0.95 de propina tarjeta

        private $totalRevenue;                 // DECIMAL(10,2)
        private $pagoCortesia;                 // DECIMAL(10,2)
        private $serviceChange;                // DECIMAL(10,2)
        private $efectivo;                     // DECIMAL(10,2)
        private $funcionario;                  // DECIMAL(10,2)
        private $yape;                         // DECIMAL(10,2)
        private $trasferencia;                 // DECIMAL(10,2)  -- Ojo: en BD es TRASFERENCIA (sin 'n')
        private $tarjeta;                      // DECIMAL(10,2)

        private $conteoFisicoCaja;             // DECIMAL(10,2)

        // === Computados en BD (también expondremos en clase) ===
        private $reportTarjMenosPropTarj;      // AS (TOTAL_REPORT_POS - TOTAL_PROP_TARJETA)
        private $efectivo1;                    // AS (EFECTIVO + FUNCIONARIO)
        private $efectivoSistemaFinal;         // AS (EFECTIVO + FUNCIONARIO - TOTAL_PROP_TARJ_095)

        private $dineroSobrante;               // DECIMAL(10,2)
        private $dineroFaltante;               // DECIMAL(10,2)
        private $observacionesCierreCaja;      // VARCHAR(100)
        private $idUsuario;                    // VARCHAR(100) (fk USUARIO.ID_USUARIO)

        // ======= Constructor =======
        public function __construct(
            $idCierreCaja = null,
            $fechaRegCaja = null,
            $totalReportPos = null,
            $totalPropTarjeta = null,
            $totalPropTarj095 = null,
            $totalRevenue = null,
            $pagoCortesia = null,
            $serviceChange = null,
            $efectivo = null,
            $funcionario = null,
            $yape = null,
            $trasferencia = null,   // coincide con la columna TRASFERENCIA de la BD
            $tarjeta = null,
            $conteoFisicoCaja = null,
            $reportTarjMenosPropTarj = null, // (calc BD)
            $efectivo1 = null,               // (calc BD)
            $efectivoSistemaFinal = null,    // (calc BD)
            $dineroSobrante = null,
            $dineroFaltante = null,
            $observacionesCierreCaja = null,
            $idUsuario = null
        ) {
            $this->idCierreCaja            = $idCierreCaja;
            $this->fechaRegCaja            = $fechaRegCaja;

            $this->totalReportPos          = $totalReportPos;
            $this->totalPropTarjeta        = $totalPropTarjeta;
            $this->totalPropTarj095        = $totalPropTarj095;

            $this->totalRevenue            = $totalRevenue;
            $this->pagoCortesia            = $pagoCortesia;
            $this->serviceChange           = $serviceChange;
            $this->efectivo                = $efectivo;
            $this->funcionario             = $funcionario;
            $this->yape                    = $yape;
            $this->trasferencia            = $trasferencia;
            $this->tarjeta                 = $tarjeta;

            $this->conteoFisicoCaja        = $conteoFisicoCaja;

            $this->reportTarjMenosPropTarj = $reportTarjMenosPropTarj;
            $this->efectivo1               = $efectivo1;
            $this->efectivoSistemaFinal    = $efectivoSistemaFinal;

            $this->dineroSobrante          = $dineroSobrante;
            $this->dineroFaltante          = $dineroFaltante;
            $this->observacionesCierreCaja = $observacionesCierreCaja;
            $this->idUsuario               = $idUsuario;
        }

        // ======= Getters y Setters =======

        public function getIdCierreCaja() { return $this->idCierreCaja; }
        public function setIdCierreCaja($v) { $this->idCierreCaja = $v; }

        public function getFechaRegCaja() { return $this->fechaRegCaja; }
        public function setFechaRegCaja($v) { $this->fechaRegCaja = $v; }

        public function getTotalReportPos() { return $this->totalReportPos; }
        public function setTotalReportPos($v) { $this->totalReportPos = $v; }

        public function getTotalPropTarjeta() { return $this->totalPropTarjeta; }
        public function setTotalPropTarjeta($v) { $this->totalPropTarjeta = $v; }

        public function getTotalPropTarj095() { return $this->totalPropTarj095; }
        public function setTotalPropTarj095($v) { $this->totalPropTarj095 = $v; }

        public function getTotalRevenue() { return $this->totalRevenue; }
        public function setTotalRevenue($v) { $this->totalRevenue = $v; }

        public function getPagoCortesia() { return $this->pagoCortesia; }
        public function setPagoCortesia($v) { $this->pagoCortesia = $v; }

        public function getServiceChange() { return $this->serviceChange; }
        public function setServiceChange($v) { $this->serviceChange = $v; }

        public function getEfectivo() { return $this->efectivo; }
        public function setEfectivo($v) { $this->efectivo = $v; }

        public function getFuncionario() { return $this->funcionario; }
        public function setFuncionario($v) { $this->funcionario = $v; }

        public function getYape() { return $this->yape; }
        public function setYape($v) { $this->yape = $v; }

        public function getTrasferencia() { return $this->trasferencia; } // coincide con TRASFERENCIA
        public function setTrasferencia($v) { $this->trasferencia = $v; }

        public function getTarjeta() { return $this->tarjeta; }
        public function setTarjeta($v) { $this->tarjeta = $v; }

        public function getConteoFisicoCaja() { return $this->conteoFisicoCaja; }
        public function setConteoFisicoCaja($v) { $this->conteoFisicoCaja = $v; }

        public function getReportTarjMenosPropTarj() { return $this->reportTarjMenosPropTarj; } // calculado BD
        public function setReportTarjMenosPropTarj($v) { $this->reportTarjMenosPropTarj = $v; }

        public function getEfectivo1() { return $this->efectivo1; } // calculado BD
        public function setEfectivo1($v) { $this->efectivo1 = $v; }

        public function getEfectivoSistemaFinal() { return $this->efectivoSistemaFinal; } // calculado BD
        public function setEfectivoSistemaFinal($v) { $this->efectivoSistemaFinal = $v; }

        public function getDineroSobrante() { return $this->dineroSobrante; }
        public function setDineroSobrante($v) { $this->dineroSobrante = $v; }

        public function getDineroFaltante() { return $this->dineroFaltante; }
        public function setDineroFaltante($v) { $this->dineroFaltante = $v; }

        public function getObservacionesCierreCaja() { return $this->observacionesCierreCaja; }
        public function setObservacionesCierreCaja($v) { $this->observacionesCierreCaja = $v; }

        public function getIdUsuario() { return $this->idUsuario; }
        public function setIdUsuario($v) { $this->idUsuario = $v; }

        // ======= Helpers (recalcular en memoria) =======

        /** Recalcula (TOTAL_REPORT_POS - TOTAL_PROP_TARJETA) */
        public function recomputeReportTarjMenosPropTarj(): void
        {
            $a = (float)($this->totalReportPos ?? 0);
            $b = (float)($this->totalPropTarjeta ?? 0);
            $this->reportTarjMenosPropTarj = round($a - $b, 2);
        }

        /** Recalcula (EFECTIVO + FUNCIONARIO) */
        public function recomputeEfectivo1(): void
        {
            $a = (float)($this->efectivo ?? 0);
            $b = (float)($this->funcionario ?? 0);
            $this->efectivo1 = round($a + $b, 2);
        }

        /** Recalcula (EFECTIVO + FUNCIONARIO - TOTAL_PROP_TARJ_095) */
        public function recomputeEfectivoSistemaFinal(): void
        {
            $a = (float)($this->efectivo ?? 0);
            $b = (float)($this->funcionario ?? 0);
            $c = (float)($this->totalPropTarj095 ?? 0);
            $this->efectivoSistemaFinal = round($a + $b - $c, 2);
        }

        /**
         * Reglas de validación de cierre (opcional):
         * - Si CONTEO_FISICO_CAJA == EFECTIVO_SISTEMA_FINAL => sobrante=0, faltante=0, obs="EL CIERRE DE CAJA ES CORRECTO"
         * - Si CONTEO_FISICO_CAJA >  EFECTIVO_SISTEMA_FINAL => sobrante=diff, faltante=0, obs="SOBRA DINERO Y EL MONTO S/ diff"
         * - Si CONTEO_FISICO_CAJA <  EFECTIVO_SISTEMA_FINAL => faltante=diff, sobrante=0, obs="ESTA FALTANDO PLATA Y EL MONTO S/ diff"
         */
        public function aplicarValidacionCierre(): void
        {
            // Asegura cálculos base
            if ($this->efectivoSistemaFinal === null) {
                $this->recomputeEfectivoSistemaFinal();
            }
            $conteo = (float)($this->conteoFisicoCaja ?? 0);
            $sist   = (float)($this->efectivoSistemaFinal ?? 0);

            if (abs($conteo - $sist) < 0.005) {
                $this->dineroSobrante = 0.00;
                $this->dineroFaltante = 0.00;
                $this->observacionesCierreCaja = 'EL CIERRE DE CAJA ES CORRECTO';
            } elseif ($conteo > $sist) {
                $diff = round($conteo - $sist, 2);
                $this->dineroSobrante = $diff;
                $this->dineroFaltante = 0.00;
                $this->observacionesCierreCaja = 'SOBRA DINERO Y EL MONTO S/ ' . number_format($diff, 2, '.', '');
            } else {
                $diff = round($sist - $conteo, 2);
                $this->dineroSobrante = 0.00;
                $this->dineroFaltante = $diff;
                $this->observacionesCierreCaja = 'ESTA FALTANDO PLATA Y EL MONTO S/ ' . number_format($diff, 2, '.', '');
            }
        }
    }
?>
