<?php
    // ==========================================
    // DAO: InformeEliminadoDAO
    // Tabla: INFORME_ELIMINADO
    // ==========================================

    require_once __DIR__ . '/../clases/InformeEliminado.php';

    class InformeEliminadoDAO {
        private $conn;

        public function __construct($conn) { 
            $this->conn = $conn; 
        }

        // ==========================
        // Helpers internos
        // ==========================
        private function toUpperClean(?string $s, int $maxLen = 100): ?string {
            if ($s === null) return null;
            $s = trim($s);
            if ($s === '') return null;
            $s = preg_replace('/\s+/u', ' ', $s);
            if (function_exists('mb_strtoupper')) {
                $s = mb_strtoupper($s, 'UTF-8');
                $s = mb_substr($s, 0, $maxLen, 'UTF-8');
            } else {
                $s = strtoupper(substr($s, 0, $maxLen));
            }
            return $s;
        }

        private function calcularTotal(int $unidad, float $precio): float {
            $u = max(0, (int)$unidad);
            $p = max(0.0, (float)$precio);
            return round($u * $p, 2);
        }

        // ==========================
        // INSERT
        // ==========================
        public function insertar(InformeEliminado $inf): bool {
            // Normaliza/limpia strings (opcional, útil si no lo haces antes)
            $idEmpleado   = $this->toUpperClean($inf->getIdEmpleado(),   100);
            $idCategoria  = $this->toUpperClean($inf->getIdCategoria(),  100);
            $idProducto   = $this->toUpperClean($inf->getIdProducto(),   100);
            $motivo       = $this->toUpperClean($inf->getMotivoEliminado(), 100);
            $idRespons    = $this->toUpperClean($inf->getIdResponsable(), 100);
            $idUsuario    = $this->toUpperClean($inf->getIdUsuario(),     100);

            $unidad = (int)$inf->getUnidadEliminado();
            $precio = (float)$inf->getPrecioProducto();

            // Calcula total (si no vino seteado)
            $total = $inf->getTotalMontoEliminado();
            if ($total === null) {
                $total = $this->calcularTotal($unidad, $precio);
            }

            // Deja FECHA_REG_ELIMINADO a GETDATE() (default)
            $sql = "INSERT INTO INFORME_ELIMINADO
                    (ID_EMPLEADO, ID_CATEGORIA, ID_PRODUCTO, 
                    UNIDAD_ELIMINADO, PRECIO_PRODUCTO, MOTIVO_ELIMINADO,
                    TOTAL_MONTO_ELIMINADO, ID_RESPONSABLE, ID_USUARIO)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";

            $params = [
                $idEmpleado,
                $idCategoria,
                $idProducto,
                $unidad,                 // INT
                $precio,                 // DECIMAL(10,2)
                $motivo,                 // Debe estar en la CHECK LIST (la BD valida)
                $total,                  // DECIMAL(12,2)
                $idRespons,
                $idUsuario
            ];

            $stmt = sqlsrv_query($this->conn, $sql, $params);
            return $stmt !== false;
        }

        // ==========================
        // LISTAR HOY (mismo patrón que tu ejemplo)
        // ==========================
        public function listarHoy(): array {
            $sql = "SELECT
                        ID_ELIMINADO,
                        FECHA_REG_ELIMINADO,
                        ID_EMPLEADO,
                        ID_CATEGORIA,
                        ID_PRODUCTO,
                        UNIDAD_ELIMINADO,
                        PRECIO_PRODUCTO,
                        MOTIVO_ELIMINADO,
                        TOTAL_MONTO_ELIMINADO,
                        ID_RESPONSABLE,
                        ID_USUARIO
                    FROM INFORME_ELIMINADO
                    WHERE CONVERT(date, FECHA_REG_ELIMINADO) = CONVERT(date, GETDATE())
                    ORDER BY ID_ELIMINADO DESC";

            $stmt = sqlsrv_query($this->conn, $sql);
            if ($stmt === false) return [];

            $rows = [];
            while ($row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
                if (isset($row['FECHA_REG_ELIMINADO']) && $row['FECHA_REG_ELIMINADO'] instanceof DateTime) {
                    $row['FECHA_REG_ELIMINADO'] = $row['FECHA_REG_ELIMINADO']->format('Y-m-d H:i:s');
                }
                $rows[] = $row;
            }
            return $rows;
        }
    }
?>