<?php
    require_once __DIR__ . '/../clases/InformeNotaCredito.php';

    class InformeNotaCreditoDAO {
        private $conn;

        // Tipos permitidos por CHECK de la tabla
        private array $tiposPermitidos = ['BOLETA', 'FUNCIONARIO', 'FACTURA'];

        public function __construct($conn) { 
            $this->conn = $conn; 
        }

        // =========================
        // Helpers de normalización
        // =========================
        private function normalizarTipo(?string $tipo): string {
            $t = strtoupper(trim((string)$tipo));
            return in_array($t, $this->tiposPermitidos, true) ? $t : 'BOLETA';
        }

        // REFF_*: 'ER023ER' y similares. Máx 10 chars, mayúsculas. Vacío -> null
        private function normalizarRef(?string $ref): ?string {
            if ($ref === null) return null;
            $v = strtoupper(trim($ref));
            if ($v === '') return null;
            return mb_substr($v, 0, 10, 'UTF-8');
        }

        private function normalizarCantidad($n): int {
            $v = is_numeric($n) ? (int)$n : 0;
            return max(0, $v);
        }

        private function normalizarMonto($m): float {
            $v = is_numeric($m) ? (float)$m : 0.0;
            // Guardamos con 2 decimales
            return round($v, 2);
        }

        private function normalizarMotivo(?string $s): string {
            $v = trim((string)$s);
            // Máx 200 (según schema)
            if (mb_strlen($v, 'UTF-8') > 200) {
                $v = mb_substr($v, 0, 200, 'UTF-8');
            }
            return $v;
        }

        // =========================
        // INSERT
        // =========================
        public function insertar(InformeNotaCredito $nota): bool {
            $tipoDoc   = $this->normalizarTipo($nota->getTipoDocumento());
            $refAnu    = $this->normalizarRef($nota->getReffAnulacion());
            $refCor    = $this->normalizarRef($nota->getReffCorreccion());
            $cant      = $this->normalizarCantidad($nota->getCantidadItems());
            $monto     = $this->normalizarMonto($nota->getMontoPago());
            $motivo    = $this->normalizarMotivo($nota->getMotivoNotaCredito());
            $idUsuario = trim((string)$nota->getIdUsuario());

            $sql = "
                INSERT INTO INFORME_NOTA_CREDITO
                    (TIPO_DOCUMENTO, REFF_ANULACION, REFF_CORRECCION,
                    CANTIDAD_ITEMS, MONTO_PAGO, MOTIVO_NOTA_CREDITO, ID_USUARIO)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ";

            $params = [
                $tipoDoc,
                $refAnu,        // puede ser null
                $refCor,        // puede ser null
                $cant,
                $monto,
                $motivo,
                $idUsuario
            ];

            $stmt = sqlsrv_query($this->conn, $sql, $params);
            if ($stmt === false) {
                error_log("❌ Error al insertar NOTA_CREDITO: " . print_r(sqlsrv_errors(), true));
                return false;
            }
            sqlsrv_free_stmt($stmt);
            return true;
        }

        
        // =========================
        // LISTAR SOLO HOY
        // =========================
        public function listarTodos(): array {
            $sql = "SELECT
                        ID_NOTA_CREDITO,
                        FECHA_REG_NOTA_CREDITO,
                        TIPO_DOCUMENTO,
                        REFF_ANULACION,
                        REFF_CORRECCION,
                        CANTIDAD_ITEMS,
                        MONTO_PAGO,
                        MOTIVO_NOTA_CREDITO,
                        ID_USUARIO
                    FROM INFORME_NOTA_CREDITO
                    WHERE CONVERT(date, FECHA_REG_NOTA_CREDITO) = CONVERT(date, GETDATE())
                    ORDER BY ID_NOTA_CREDITO DESC";

            $stmt = sqlsrv_query($this->conn, $sql);
            if ($stmt === false) return [];

            $rows = [];
            while ($row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
                if (isset($row['FECHA_REG_NOTA_CREDITO']) && $row['FECHA_REG_NOTA_CREDITO'] instanceof DateTime) {
                    $row['FECHA_REG_NOTA_CREDITO'] = $row['FECHA_REG_NOTA_CREDITO']->format('Y-m-d H:i:s');
                }
                $rows[] = $row;
            }
            sqlsrv_free_stmt($stmt);
            return $rows;
        }
    }
?>