<?php
    // /backend/procesos/Guardar_Informe_Gasto.php
    session_start();
    require_once __DIR__ . '/../database/Conexion_SQL_Server.php';
    require_once __DIR__ . '/../clases/InformeGasto.php';
    require_once __DIR__ . '/../dao/InformeGastoDAO.php';

    date_default_timezone_set('America/Lima');

    /**
     * Convierte a MAYÚSCULAS con limpieza de espacios.
     * - Colapsa espacios múltiples a uno.
     * - Usa mb_strtoupper si está disponible (UTF-8).
     * - Recorta a longitud máxima con mb_substr si existe.
     */
    function to_upper_clean(?string $s, int $maxLen): ?string {
        if ($s === null) return null;
        $s = trim($s);
        if ($s === '') return null;
        $s = preg_replace('/\s+/u', ' ', $s); // colapsa espacios
        $s = function_exists('mb_strtoupper') ? mb_strtoupper($s, 'UTF-8') : strtoupper($s);
        if (function_exists('mb_substr')) {
            $s = mb_substr($s, 0, $maxLen, 'UTF-8');
        } else {
            $s = substr($s, 0, $maxLen);
        }
        return $s;
    }

    /** Convierte texto tipo "12,50" a float (>=0). */
    function parse_amount($v): float {
        if ($v === null) return 0.0;
        $v = str_replace(',', '.', (string)$v);
        $n = (float)$v;
        if (!is_finite($n)) $n = 0.0;
        return max(0.0, round($n, 2));
    }

    try {
        // ============================
        // Seguridad básica
        // ============================
        if (!isset($_SESSION['id_usuario'])) {
            header('Location: ../../frontend/Login_LG.html?error=' . urlencode('Sesion expirada')); exit;
        }
        if (($_SESSION['rol'] ?? '') !== 'CAJA') {
            header('Location: ../../frontend/home.php?error=' . urlencode('sin_permiso')); exit;
        }
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: ../../frontend/app_Caja.php?err=' . urlencode('Método no permitido') . '&tab=igasto'); exit;
        }

        $idUsuario = $_SESSION['id_usuario'];

        // ============================
        // Entradas
        // ============================
        $persona = to_upper_clean($_POST['ig_persona'] ?? null, 100); // requerido
        $motivo  = to_upper_clean($_POST['ig_motivo']  ?? null, 200); // opcional
        $monto   = parse_amount($_POST['ig_monto'] ?? null);

        // ============================
        // Validaciones mínimas
        // ============================
        if (!$persona) {
            header('Location: ../../frontend/app_Caja.php?err=' . urlencode('Ingrese la persona que entrega') . '&tab=igasto'); exit;
        }
        if (mb_strlen($persona, 'UTF-8') > 100) {
            header('Location: ../../frontend/app_Caja.php?err=' . urlencode('Persona excede 100 caracteres') . '&tab=igasto'); exit;
        }
        if ($motivo !== null && mb_strlen($motivo, 'UTF-8') > 200) {
            header('Location: ../../frontend/app_Caja.php?err=' . urlencode('Motivo excede 200 caracteres') . '&tab=igasto'); exit;
        }
        if ($monto <= 0) {
            header('Location: ../../frontend/app_Caja.php?err=' . urlencode('Monto inválido (debe ser > 0)') . '&tab=igasto'); exit;
        }

        // ============================
        // Modelo
        // ============================
        $model = new InformeGasto(
            $persona,  // PERSONA_ENTREGADA (en mayúsculas)
            $motivo,   // MOTIVO_GASTO (en mayúsculas o NULL)
            $monto,    // MONTO_GASTO
            $idUsuario // ID_USUARIO (sesión)
        );

        // ============================
        // Persistencia
        // ============================
        $dao = new InformeGastoDAO($conn);
        $idInsert = $dao->insertar($model);

        // ============================
        // Resultado / Redirección
        // ============================
        if ($idInsert !== null) {
        header('Location: ../../frontend/app_Caja.php?ok=' . urlencode('Gasto registrado correctamente') . '&tab=igasto');
        } else {
            header('Location: ../../frontend/app_Caja.php?err=' . urlencode('No se pudo registrar el gasto') . '&tab=igasto');
        }
        exit;

    } catch (Throwable $e) {
        error_log('Guardar_Informe_Gasto: ' . $e->getMessage());
        header('Location: ../../frontend/app_Caja.php?err=' . urlencode('Error interno') . '&tab=igasto'); 
        exit;
    }
?>