<?php
session_start();

if (!isset($_SESSION['id_usuario'])) {
  header('Location: Login_LG.html?error=Sesion expirada');
  exit;
}
if (($_SESSION['rol'] ?? '') !== 'ADMIN') {
  header('Location: home.php?error=sin_permiso');
  exit;
}

$idUsuario    = $_SESSION['id_usuario'] ?? '';
$rol          = $_SESSION['rol'] ?? '';
$nombreCajero = $_SESSION['nombre_cajero'] ?? $idUsuario;
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
  <title>Administrador – Importar a Cierre de Caja</title>

  <!-- Bootstrap -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <!-- Bootstrap Icons -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
  <!-- Tus estilos -->
  <link rel="stylesheet" href="../css/Estilos_appAdmin.css">
</head>
<body>

  <!-- Overlay de ambiente -->
  <div class="bg-overlay"></div>

  <!-- Header fijo estilizado -->
  <header class="header-lg">
    <div class="header-left">
      <div class="header-avatar"><i class="bi bi-person-fill"></i></div>
      <div class="header-user">
        <span class="header-name"><?= htmlspecialchars($nombreCajero, ENT_QUOTES, 'UTF-8'); ?></span>
        <span class="header-role"><?= htmlspecialchars($rol, ENT_QUOTES, 'UTF-8'); ?></span>
      </div>
    </div>

    <div class="header-actions">
      <form action="../backend/procesos/seguridad/logout.php" method="post" class="m-0"
            onsubmit="return confirm('¿Deseas cerrar sesión?');">
        <button type="submit" class="btn-hdr btn-hdr--ghost">
          <i class="bi bi-box-arrow-right"></i> Cerrar sesión
        </button>
      </form>
    </div>
  </header>

  <!-- Contenido -->
  <main class="container container-guardiana">
    <div class="panel-card">
      <div class="d-flex justify-content-between align-items-center mb-2">
        <h1 class="h-title m-0">Importar datos (Python → Excel destino)</h1>
        <span class="subtle">Rol actual: <strong><?= htmlspecialchars($rol, ENT_QUOTES, 'UTF-8'); ?></strong></span>
      </div>
      <div class="rule-rose"></div>

      <!-- CUADRO: Importador -->
      <section class="import-card mb-3" id="importCard">
        <!-- 🔔 Alertas internas (arriba del cuadro) -->
        <div class="card-alerts" id="cardAlerts" aria-live="polite" aria-atomic="true"></div>

        <div class="import-card__head">
          <div class="import-card__icon">
            <i class="bi bi-file-earmark-excel"></i>
          </div>

          <div>
            <h3 class="import-card__title">Importar al archivo destino</h3>
            <p class="import-card__subtitle">Selecciona el archivo de origen y ejecuta el proceso.</p>
            <!-- ❌ Chips de destino/hoja eliminados por privacidad -->
          </div>
        </div>

        <div class="import-card__body">
          <?php
            $excels_dir = realpath(__DIR__ . '/../excels');
            $archivos_excel = [];
            if ($excels_dir && is_dir($excels_dir)) {
              foreach (scandir($excels_dir) as $f) {
                if (preg_match('/\.(xlsx|xls)$/i', $f)) $archivos_excel[] = $f;
              }
              sort($archivos_excel);
            }
          ?>

          <form id="frmImport" class="import-grid">
            <div class="import-field">
              <label class="form-label">Selecciona archivo de origen</label>
              <select name="origen_nombre" class="form-select" required>
                <option value="" disabled selected>— Selecciona un archivo —</option>
                <?php foreach ($archivos_excel as $f): ?>
                  <option value="<?= htmlspecialchars($f, ENT_QUOTES) ?>"><?= htmlspecialchars($f) ?></option>
                <?php endforeach; ?>
              </select>
              <div class="import-hint">
                Asegúrate de que la hoja <code>Reports</code> exista dentro del archivo de origen.
              </div>
            </div>

            <div class="d-grid align-content-end">
              <button type="submit" class="btn btn-cta" id="btnRun">
                <i class="bi bi-play-fill"></i> Ejecutar
              </button>
            </div>
          </form>

          <div id="resultado" class="mt-3"></div>
        </div>

        <div class="import-card__footer">
          Consejo: si el proceso tarda, revisa permisos del sistema y el antivirus.
        </div>
      </section>
      <!-- /CUADRO: Importador -->

      <!-- CUADRO: Cierre de caja -->
      <section class="import-card mb-3">
        <div class="import-card__head">
          <div class="import-card__icon">
            <i class="bi bi-cash-coin"></i>
          </div>
          <div>
            <h3 class="import-card__title">Cierre de caja</h3>
            <p class="import-card__subtitle">Previsualiza las cifras y confirma el cierre del día.</p>
          </div>
        </div>

        <div class="import-card__body">
          <div class="row g-3">
            <div class="col-md-4">
              <label class="form-label">Conteo físico (S/)</label>
              <input id="cc_conteo" type="number" step="0.01" min="0" class="form-control" placeholder="0.00" required>
            </div>
            <div class="col-md-8 d-flex align-items-end gap-2">
              <button type="button" class="btn btn-outline-primary" id="cc_preview">
                <i class="bi bi-eye"></i> Previsualizar
              </button>
              <button type="button" class="btn btn-cta" id="cc_confirmar" disabled>
                <i class="bi bi-check2-circle"></i> Confirmar e Insertar
              </button>
            </div>
          </div>

          <div id="cc_panel" class="mt-4 d-none">
            <div class="row g-3">
              <div class="col-md-4">
                <div class="card shadow-sm h-100">
                  <div class="card-header fw-semibold">Montos desde Excel</div>
                  <table class="table mb-0 small" id="cc_tbl_excel"></table>
                </div>
              </div>
              <div class="col-md-4">
                <div class="card shadow-sm h-100">
                  <div class="card-header fw-semibold">Totales desde SQL</div>
                  <table class="table mb-0 small" id="cc_tbl_sql"></table>
                </div>
              </div>
              <div class="col-md-4">
                <div class="card shadow-sm h-100">
                  <div class="card-header fw-semibold">Cálculos y validación</div>
                  <table class="table mb-0 small" id="cc_tbl_calc"></table>
                </div>
              </div>
            </div>
          </div>
        </div>

        <div class="import-card__footer">
          El cierre solo se puede registrar <strong>una vez por día</strong>. Si ya existe, verás un error.
        </div>
      </section>
      <!-- /CUADRO: Cierre de caja -->

    </div>
  </main>

  <script>
    const form   = document.getElementById('frmImport');
    const btn    = document.getElementById('btnRun');
    const out    = document.getElementById('resultado');
    const alerts = document.getElementById('cardAlerts');

    // Alertas dentro de la tarjeta (auto-close + botón cerrar)
    function showCardAlert(type, text, timeoutMs = 4000){
      // type: 'info' | 'ok' | 'warn' | 'error'
      const map = {
        info:  {cls:'card-alert--info',  icon:'bi bi-hourglass-split'},
        ok:    {cls:'card-alert--ok',    icon:'bi bi-check-circle-fill'},
        warn:  {cls:'card-alert--warn',  icon:'bi bi-exclamation-triangle-fill'},
        error: {cls:'card-alert--error', icon:'bi bi-x-octagon-fill'}
      };
      const {cls, icon} = map[type] ?? map.info;

      const el = document.createElement('div');
      el.className = `card-alert ${cls}`;
      el.setAttribute('role','alert');
      el.innerHTML = `
        <i class="icon ${icon}"></i>
        <div class="msg">${text}</div>
        <button type="button" class="btn-close" aria-label="Cerrar">&times;</button>
      `;
      alerts.prepend(el);

      const closer = ()=>{ el.style.opacity='0'; el.style.transform='translateY(-6px)'; setTimeout(()=> el.remove(), 220); };
      el.querySelector('.btn-close').addEventListener('click', closer);
      if (timeoutMs>0) setTimeout(closer, timeoutMs);
      return el;
    }

    // ===== Importador =====
    form.addEventListener('submit', async (e)=>{
      e.preventDefault();
      btn.disabled = true;

      // Mensaje “Ejecutando…”
      const running = showCardAlert('info', '⏳ Ejecutando Python…', 0);

      try {
        const fd = new FormData(form);
        // Hoja destino (si tu backend la usa)
        fd.append('hoja_destino', 'IMPORTACION_ORACLE_LG');

        const r  = await fetch('/Proyecto_Guardiana/backend/procesos/run_import_excel.php', {
          method: 'POST',
          body: fd
        });
        const j = await r.json();

        running.remove();

        if (!j.ok) {
          showCardAlert('error', `❌ ${j.error || 'Error desconocido'}`);
          out.innerHTML = j.salida
            ? '<div class="table-wrap mt-2"><pre class="small p-2 m-0">'+ (Array.isArray(j.salida)? j.salida.join('\n'): j.salida) +'</pre></div>'
            : '';
          return;
        }

        showCardAlert('ok', '✅ Importación OK');
        out.innerHTML = ''; // limpia logs si no quieres mostrarlos
      } catch (err) {
        running.remove();
        showCardAlert('error', `❌ ${err.message}`);
      } finally {
        btn.disabled = false;
      }
    });

    // ===== Cierre de Caja: preview + confirmar =====
    (function(){
      const fmt = n => (Number(n)||0).toFixed(2);

      const $conteo   = document.getElementById('cc_conteo');
      const $preview  = document.getElementById('cc_preview');
      const $confirm  = document.getElementById('cc_confirmar');
      const $panel    = document.getElementById('cc_panel');
      const $tblExcel = document.getElementById('cc_tbl_excel');
      const $tblSql   = document.getElementById('cc_tbl_sql');
      const $tblCalc  = document.getElementById('cc_tbl_calc');

      function renderTableExcel(mp){
        $tblExcel.innerHTML = `
          <tbody>
            <tr><th>Total</th><td class="text-end">S/ ${fmt(mp.total)}</td></tr>
            <tr><th>Efectivo</th><td class="text-end">S/ ${fmt(mp.efectivo)}</td></tr>
            <tr><th>Funcionario</th><td class="text-end">S/ ${fmt(mp.funcionario)}</td></tr>
            <tr><th>Pago Cortesía</th><td class="text-end">S/ ${fmt(mp.pago_cortesia)}</td></tr>
            <tr><th>Yape</th><td class="text-end">S/ ${fmt(mp.yape)}</td></tr>
            <tr><th>Transferencia</th><td class="text-end">S/ ${fmt(mp.transferencia)}</td></tr>
            <tr><th>Tarjeta</th><td class="text-end">S/ ${fmt(mp.tarjeta)}</td></tr>
            <tr><th>Cargos por servicio</th><td class="text-end">S/ ${fmt(mp.service_change)}</td></tr>
          </tbody>`;
      }
      function renderTableSql(sql){
        $tblSql.innerHTML = `
          <tbody>
            <tr><th>Total Report POS</th><td class="text-end">S/ ${fmt(sql.TOTAL_REPORT_POS)}</td></tr>
            <tr><th>Total Propina Tarj. (bruta)</th><td class="text-end">S/ ${fmt(sql.TOTAL_PROP_TARJETA)}</td></tr>
            <tr><th>Total Propina Tarj. 0.95</th><td class="text-end">S/ ${fmt(sql.TOTAL_PROP_TARJ_095)}</td></tr>
          </tbody>`;
      }
      function renderTableCalc(c){
        $tblCalc.innerHTML = `
          <tbody>
            <tr><th>Conteo físico</th><td class="text-end">S/ ${fmt(c.CONTEO_FISICO_CAJA)}</td></tr>
            <tr><th>Report POS - Prop. Tarjeta</th><td class="text-end">S/ ${fmt(c.REPORT_TARJ_MENOS_PROP_TARJ)}</td></tr>
            <tr><th>Efectivo + Funcionario</th><td class="text-end">S/ ${fmt(c.EFECTIVO_1)}</td></tr>
            <tr><th>Efectivo Sistema Final</th><td class="text-end fw-bold">S/ ${fmt(c.EFECTIVO_SISTEMA_FINAL)}</td></tr>
            <tr><th>Sobrante</th><td class="text-end ${c.DINERO_SOBRANTE>0?'text-danger':'text-muted'}">S/ ${fmt(c.DINERO_SOBRANTE)}</td></tr>
            <tr><th>Faltante</th><td class="text-end ${c.DINERO_FALTANTE>0?'text-warning':'text-muted'}">S/ ${fmt(c.DINERO_FALTANTE)}</td></tr>
            <tr><th>Observación</th><td class="text-end">${c.OBSERVACION}</td></tr>
          </tbody>`;
      }

      $preview?.addEventListener('click', async ()=>{
        const val = parseFloat($conteo.value||0);
        if (isNaN(val) || val<0) { alert('Ingrese conteo físico válido.'); return; }

        $preview.disabled = true; $preview.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Consultando...';
        try{
          const r = await fetch('/Proyecto_Guardiana/backend/procesos/Preview_Cierre_Caja.php?conteo='+encodeURIComponent(val));
          const j = await r.json();
          if(!j.ok) throw new Error(j.error||'Error desconocido');

          renderTableExcel(j.data.excel_medios);
          renderTableSql(j.data.sql_totales);
          renderTableCalc(j.data.calculados);

          $panel.classList.remove('d-none');
          $confirm.disabled = false;
        }catch(e){
          alert('No se pudo previsualizar: '+e.message);
          $panel.classList.add('d-none');
          $confirm.disabled = true;
        }finally{
          $preview.disabled = false; $preview.innerHTML = '<i class="bi bi-eye"></i> Previsualizar';
        }
      });

      // ⬇️⬇️ LISTENER ACTUALIZADO (tolerante a respuestas no JSON puras) ⬇️⬇️
      $confirm?.addEventListener('click', async ()=>{
      const val = parseFloat($conteo.value||0);
      if (!confirm('¿Confirmar e insertar el cierre?')) return;

      $confirm.disabled = true;
      $confirm.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>Insertando...';

      try {
        const r = await fetch('/Proyecto_Guardiana/backend/procesos/Insertar_Cierre_Caja.php', {
          method: 'POST',
          headers: {'Content-Type':'application/x-www-form-urlencoded'},
          body: 'conteo=' + encodeURIComponent(val)
        });

        const raw = await r.text();
        let j;
        try {
          j = JSON.parse(raw);
        } catch {
          throw new Error('Respuesta no válida del servidor: ' + raw.substring(0, 200));
        }

        if (j && j.ok) {
          // ✅ Éxito: Mostrar ID y limpiar todo
          showCardAlert('ok', '✅ Cierre registrado correctamente.');

          // 🧹 Limpieza visual
          $conteo.value = '';                                // limpia el input
          document.getElementById('cc_panel').classList.add('d-none'); // oculta panel
          document.getElementById('cc_confirmar').disabled = true;     // bloquea botón
          document.getElementById('cc_tbl_excel').innerHTML = '';      // limpia tablas
          document.getElementById('cc_tbl_sql').innerHTML   = '';
          document.getElementById('cc_tbl_calc').innerHTML  = '';

          return;
        }

        const msg = (j && j.error) ? j.error : 'No se pudo insertar';
        throw new Error(msg);

      } catch (e) {
        alert('Error al insertar: ' + (e.message || 'desconocido'));
      } finally {
        $confirm.disabled = false;
        $confirm.innerHTML = '<i class="bi bi-check2-circle"></i> Confirmar e Insertar';
      }
    });

      // ⬆️⬆️ FIN LISTENER ACTUALIZADO ⬆️⬆️

    })();
  </script>
</body>
</html>
