<?php
    // ... cabecera igual
    require_once __DIR__ . '/../clases/InformeDescuento.php';

    class InformeDescuentoDAO {
        private $conn;

        // ⚙️ Config: paso de redondeo para el total final (0.10 -> 10 céntimos)
        private float $pasoRedondeoFinal = 0.10; // puedes poner 0.05, 0.10, etc.

        public function __construct($conn) { $this->conn = $conn; }

        private function calcularDescuentoMonetario($montoBase, $porcentaje): float {
            $m = is_numeric($montoBase) ? (float)$montoBase : 0.0;
            $p = is_numeric($porcentaje) ? max(0.0, min(100.0, (float)$porcentaje)) : 0.0;
            return round($m * ($p / 100.0), 1); // 1 decimal
        }

        private function calcularTotalFinal($montoBase, $descuentoMonetario): float {
            $m = is_numeric($montoBase) ? (float)$montoBase : 0.0;
            $d = is_numeric($descuentoMonetario) ? (float)$descuentoMonetario : 0.0;
            return $m - $d; // primero sin redondeo
        }

        // 🔁 Redondeo del total a un "paso" (nearest). Ej: 0.10 -> 3.45 ⇒ 3.50
        private function redondearAlPaso(float $valor, float $paso = 0.10): float {
            if ($paso <= 0) return round($valor, 2);
            $resultado = round($valor / $paso) * $paso; // nearest
            return round($resultado, 2); // guardamos con 2 decimales
        }

        // ===== INSERT =====
        public function insertar(InformeDescuento $inf): bool {
            $monto = $inf->getMontoDescuento();      // base
            $porc  = $inf->getPorcentajeDescuento(); // %

            $descMonetario = $this->calcularDescuentoMonetario($monto, $porc);     // 1 dec.
            $totalSinPaso  = $this->calcularTotalFinal($monto, $descMonetario);    // sin redondeo
            $totalFinal    = $this->redondearAlPaso($totalSinPaso, $this->pasoRedondeoFinal); // a 0.10

            // 👇 Asegura entero (DECIMAL(10,0))
            $check = (int)$inf->getCheckDescuento();

            $sql = "INSERT INTO INFORME_DESCUENTO
                    (CHECK_DESCUENTO, NOMBRE_PERSONA, MOTIVO_DESCUENTO,
                    MONTO_DESCUENTO, PORCENTAJE_DESCUENTO, DESCUENTO_PORCENTAJE, TOTAL_MONTO_DESCUENTO,
                    ID_RESPONSABLE, ID_USUARIO)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";

            $params = [
                $check,                          // ✅ entero para DECIMAL(10,0)
                $inf->getNombrePersona(),
                $inf->getMotivoDescuento(),
                $monto,
                $porc,
                $descMonetario,                  // 1 decimal
                $totalFinal,                     // redondeado a 0.10 y 2 decimales
                $inf->getIdResponsable(),
                $inf->getIdUsuario()
            ];

            $stmt = sqlsrv_query($this->conn, $sql, $params);
            return $stmt !== false;
        }


        
        // ===== LISTAR HOY (igual que tenías) =====
        public function listarTodos(): array {
            $sql = "SELECT
                        ID_DESCUENTO,
                        FECHA_REG_DESCUENTO,
                        CHECK_DESCUENTO,
                        NOMBRE_PERSONA,
                        MOTIVO_DESCUENTO,
                        MONTO_DESCUENTO,
                        PORCENTAJE_DESCUENTO,
                        DESCUENTO_PORCENTAJE,                    -- 1 dec.
                        TOTAL_MONTO_DESCUENTO,                   -- 2 dec., redondeado al paso
                        (MONTO_DESCUENTO - TOTAL_MONTO_DESCUENTO) AS MONTO_DESCONTADO_CALC,
                        ID_RESPONSABLE,
                        ID_USUARIO
                    FROM INFORME_DESCUENTO
                    WHERE CONVERT(date, FECHA_REG_DESCUENTO) = CONVERT(date, GETDATE())
                    ORDER BY ID_DESCUENTO DESC";

            $stmt = sqlsrv_query($this->conn, $sql);
            if ($stmt === false) return [];

            $rows = [];
            while ($row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
                if (isset($row['FECHA_REG_DESCUENTO']) && $row['FECHA_REG_DESCUENTO'] instanceof DateTime) {
                    $row['FECHA_REG_DESCUENTO'] = $row['FECHA_REG_DESCUENTO']->format('Y-m-d H:i:s');
                }
                $rows[] = $row;
            }
            return $rows;
        }
    }
?>