<?php
    // backend/dao/CierreCajaDAO.php
    require_once __DIR__ . '/../database/Conexion_DataBase.php'; // define $pdo
    require_once __DIR__ . '/../clases/CierreCaja.php';

    class CierreCajaDAO {
        /** @var PDO */
        private PDO $db;

        public function __construct(?PDO $conn = null) {
            // si no te pasan conexión, usa la global creada en Conexion_DataBase.php
            $this->db = $conn ?? $GLOBALS['pdo'];
        }

        // ================================
        // ¿Existe cierre global HOY?
        // ================================
        private function existeCierreHoy(): bool {
            $sql = "SELECT 1
                    FROM CIERRE_CAJA
                    WHERE DATE(FECHA_CIERRE) = CURDATE()
                    LIMIT 1";
            return (bool)$this->db->query($sql)->fetchColumn();
        }

        // ================================
        // Insertar Cierre (solo uno por día)
        // ================================
        public function insertarCierre(CierreCaja $cierre) {
            try {
                if ($this->existeCierreHoy()) {
                    // Ya hay cierre hoy (global)
                    return false; // o retorna "YA_EXISTE" si prefieres
                }

                $sql = "INSERT INTO CIERRE_CAJA (
                            MONTO_APERTURA_DIA,
                            TOTAL_EFECTIVO_DIA,
                            TOTAL_YAPE_PLIN_DIA,
                            TOTAL_TARJETA_DIA,
                            TOTAL_REPORTE_DIA,
                            TOTAL_RETIROS_DIA,
                            TOTAL_INGRESOS_DIA,
                            EFECTIVO_SISTEMA_FINAL,
                            CONTEO_FISICO_CAJA,
                            DINERO_SOBRANTE,
                            DINERO_FALTANTE,
                            OBSERVACION_CIERRE,
                            ID_USUARIO
                        ) VALUES (
                            :monto_apertura,
                            :total_efectivo,
                            :total_yape_plin,
                            :total_tarjeta,
                            :total_reporte,
                            :total_retiros,
                            :total_ingresos,
                            :efectivo_final,
                            :conteo_fisico,
                            :sobrante,
                            :faltante,
                            :observacion,
                            :usuario
                        )";

                $stmt = $this->db->prepare($sql);
                $ok = $stmt->execute([
                    ':monto_apertura' => $cierre->getMontoAperturaDia(),
                    ':total_efectivo' => $cierre->getTotalEfectivoDia(),
                    ':total_yape_plin'=> $cierre->getTotalYapePlinDia(),
                    ':total_tarjeta'  => $cierre->getTotalTarjetaDia(),
                    ':total_reporte'  => $cierre->getTotalReporteDia(),
                    ':total_retiros'  => $cierre->getTotalRetirosDia(),
                    ':total_ingresos' => $cierre->getTotalIngresosDia(),
                    ':efectivo_final' => $cierre->getEfectivoSistemaFinal(),
                    ':conteo_fisico'  => $cierre->getConteoFisicoCaja(),
                    ':sobrante'       => $cierre->getDineroSobrante(),
                    ':faltante'       => $cierre->getDineroFaltante(),
                    ':observacion'    => $cierre->getObservacionCierre(),
                    ':usuario'        => $cierre->getIdUsuario()
                ]);

                return $ok ? true : false;

            } catch (PDOException $e) {
                // Si agregas una restricción única por día y falla, MySQL lanza 1062
                if ((int)$e->errorInfo[1] === 1062) {
                    return false; // o "YA_EXISTE"
                }
                // error_log($e->getMessage());
                return false;
            }
        }

        // (útil) Listar cierre(s) de HOY
        public function listarDeHoy(): array {
            $sql = "SELECT ID_CIERRE, FECHA_CIERRE, MONTO_APERTURA_DIA, TOTAL_EFECTIVO_DIA,
                        TOTAL_YAPE_PLIN_DIA, TOTAL_TARJETA_DIA, TOTAL_REPORTE_DIA,
                        TOTAL_RETIROS_DIA, TOTAL_INGRESOS_DIA, EFECTIVO_SISTEMA_FINAL,
                        CONTEO_FISICO_CAJA, DINERO_SOBRANTE, DINERO_FALTANTE,
                        OBSERVACION_CIERRE, ID_USUARIO
                    FROM CIERRE_CAJA
                    WHERE DATE(FECHA_CIERRE) = CURDATE()
                    ORDER BY ID_CIERRE DESC";
            return $this->db->query($sql)->fetchAll(PDO::FETCH_ASSOC) ?: [];
        }
    }
?>