<?php
    // backend/dao/ReporteVentaDAO.php
    require_once __DIR__ . '/../database/Conexion_DataBase.php'; // define $pdo

    class ReporteVentaDAO {
        /** @var PDO */
        private PDO $db;

        public function __construct(?PDO $conn = null) {
            $this->db = $conn ?? $GLOBALS['pdo'];
        }

        /** Retorna el ID del reporte global de HOY si existe, si no existe retorna 0. */
        public function existeReporteHoy(): int {
            $sql = "SELECT ID_REPORTE
                    FROM REPORTE_VENTA
                    WHERE DATE(FECHA_REPORT_VENTA) = CURDATE()
                    ORDER BY ID_REPORTE DESC
                    LIMIT 1";
            $id = $this->db->query($sql)->fetchColumn();
            return $id ? (int)$id : 0;
        }

        /** SUMA la(s) apertura(s) global(es) de HOY */
        private function obtenerAperturaDeHoyGlobal(): float {
            $sql = "SELECT IFNULL(SUM(MONTO_APERTURA), 0) AS APERTURA_TOTAL
                    FROM APERTURA_CAJA
                    WHERE DATE(FECHA_APERTURA) = CURDATE()";
            $val = $this->db->query($sql)->fetchColumn();
            return (float)($val ?? 0.0);
        }

        /**
         * Genera REPORTE_VENTA GLOBAL una sola vez por día.
         * - Si ya existe, retorna su ID y NO inserta.
         * - Si no existe, lo calcula e inserta (retorna ID).
         *
         * Asume columnas:
         *   ID_REPORTE (AUTO_INCREMENT),
         *   FECHA_REPORT_VENTA (DEFAULT CURRENT_TIMESTAMP),
         *   MONTO_APERTURA, TOTAL_EFECTIVO, TOTAL_YAPE_PLIN, TOTAL_TARJETA,
         *   TOTAL_DESCUENTO, TOTAL_REPORTE, ID_USUARIO
         */
        public function generarDesdeVentasDeHoyGLOBAL($montoApertura = null, $idUsuarioGenera = 'SISTEMA') {
            // 0) ¿ya existe?
            $idExistente = $this->existeReporteHoy();
            if ($idExistente > 0) {
                return $idExistente;
            }

            // 1) Apertura (si no llega, se calcula)
            if ($montoApertura === null || $montoApertura === '') {
                $montoApertura = $this->obtenerAperturaDeHoyGlobal();
            } else {
                $montoApertura = (float)$montoApertura;
            }

            // 2) Totales globales del día
            $sqlTot = "SELECT
                            IFNULL(SUM(CASE WHEN METODO_PAGO = 'EFECTIVO'  THEN TOTAL_VENTA ELSE 0 END), 0) AS TOTAL_EFECTIVO,
                            IFNULL(SUM(CASE WHEN METODO_PAGO = 'YAPE_PLIN' THEN TOTAL_VENTA ELSE 0 END), 0) AS TOTAL_YAPE_PLIN,
                            IFNULL(SUM(CASE WHEN METODO_PAGO = 'TARJETA'   THEN TOTAL_VENTA ELSE 0 END), 0) AS TOTAL_TARJETA,
                            IFNULL(SUM(DESCUENTO_PRODUCTO), 0) AS TOTAL_DESCUENTO
                    FROM VENTA
                    WHERE DATE(FECHA_REGISTRO_VENTA) = CURDATE()";
            $row = $this->db->query($sqlTot)->fetch(PDO::FETCH_ASSOC) ?: [];

            $efectivo  = (float)($row['TOTAL_EFECTIVO']  ?? 0);
            $yapePlin  = (float)($row['TOTAL_YAPE_PLIN'] ?? 0);
            $tarjeta   = (float)($row['TOTAL_TARJETA']   ?? 0);
            $descuento = (float)($row['TOTAL_DESCUENTO'] ?? 0);

            // 3) Total del reporte = suma de métodos (NO restar descuentos otra vez)
            $totalRep  = $efectivo + $yapePlin + $tarjeta;

            // 4) INSERT y devolver ID (MySQL: lastInsertId)
            $sqlIns = "INSERT INTO REPORTE_VENTA
                        (MONTO_APERTURA, TOTAL_EFECTIVO, TOTAL_YAPE_PLIN, TOTAL_TARJETA,
                        TOTAL_DESCUENTO, TOTAL_REPORTE, ID_USUARIO)
                    VALUES
                        (:apertura, :efec, :yape, :tarj, :desc, :total, :usr)";
            $stmt = $this->db->prepare($sqlIns);
            $ok = $stmt->execute([
                ':apertura' => $montoApertura,
                ':efec'     => $efectivo,
                ':yape'     => $yapePlin,
                ':tarj'     => $tarjeta,
                ':desc'     => $descuento,
                ':total'    => $totalRep,
                ':usr'      => $idUsuarioGenera
            ]);

            if (!$ok) return false;

            // En MySQL no hay OUTPUT INSERTED; usamos lastInsertId()
            $newId = (int)$this->db->lastInsertId();
            return $newId > 0 ? $newId : false;
        }
    }
?>