<?php
// backend/dao/VentaDAO.php
require_once __DIR__ . '/../database/Conexion_DataBase.php'; // define $pdo
require_once __DIR__ . '/../clases/Venta.php';

class VentaDAO {
    /** @var PDO */
    private PDO $db;

    public function __construct(?PDO $conn = null) {
        $this->db = $conn ?? $GLOBALS['pdo'];
    }

    /**
     * Inserta una venta tomando el precio actual desde PRODUCTO.
     * Calcula TOTAL_VENTA = (CANTIDAD * PRECIO_PRODUCTO) - DESCUENTO (no negativo).
     * Si envías FECHA_REGISTRO_VENTA en la entidad, la usa; si no, se asume DEFAULT de la tabla.
     */
    public function insertarUsandoProducto(Venta $venta): bool {
        // Validar método de pago permitido
        $permitidos = ['EFECTIVO','YAPE_PLIN','TARJETA'];
        $metodo = strtoupper(trim($venta->getMetodoPago()));
        if (!in_array($metodo, $permitidos, true)) {
            return false;
        }

        try {
            // 1) Obtener precio desde PRODUCTO
            $sqlPrecio = "SELECT PRECIO_PRODUCTO FROM PRODUCTO WHERE ID_PRODUCTO = :id LIMIT 1";
            $stmtP = $this->db->prepare($sqlPrecio);
            $stmtP->execute([':id' => $venta->getIdProducto()]);
            $rowP = $stmtP->fetch(PDO::FETCH_ASSOC);
            if (!$rowP) return false; // producto no existe

            $precio = (float)$rowP['PRECIO_PRODUCTO'];

            // 2) Calcular total
            $cantidad  = (int)$venta->getCantidad();
            $descuento = (float)$venta->getDescuentoProducto();
            $total     = ($cantidad * $precio) - $descuento;
            if ($total < 0) { $total = 0; }

            // 3) Construir INSERT dinámico si trae fecha
            $sqlCols = "ID_PRODUCTO, CANTIDAD, MONTO_PRODUCTO, DESCUENTO_PRODUCTO, METODO_PAGO, TOTAL_VENTA, ID_USUARIO";
            $place   = ":prod, :cant, :monto, :desc, :met, :total, :usr";
            $params  = [
                ':prod'  => $venta->getIdProducto(),
                ':cant'  => $cantidad,
                ':monto' => $precio,
                ':desc'  => $descuento,
                ':met'   => $metodo,
                ':total' => $total,
                ':usr'   => $venta->getIdUsuario()
            ];

            if ($venta->getFechaRegistroVenta() !== null && $venta->getFechaRegistroVenta() !== '') {
                $sqlCols = "FECHA_REGISTRO_VENTA, " . $sqlCols;
                $place   = ":fecha, " . $place;
                $params[':fecha'] = $venta->getFechaRegistroVenta(); // 'YYYY-MM-DD HH:MM:SS'
            }

            $sql = "INSERT INTO VENTA ($sqlCols) VALUES ($place)";
            $stmt = $this->db->prepare($sql);
            return $stmt->execute($params);

        } catch (PDOException $e) {
            // error_log($e->getMessage());
            return false;
        }
    }

    /**
     * Lista las ventas del día (HOY).
     * ORDER BY FECHA_REGISTRO_VENTA DESC, ID_VENTA DESC
     */
    public function listarTodos(): array {
        try {
            $sql = "SELECT
                        FECHA_REGISTRO_VENTA,
                        ID_VENTA,
                        ID_PRODUCTO,
                        CANTIDAD,
                        MONTO_PRODUCTO,
                        DESCUENTO_PRODUCTO,
                        METODO_PAGO,
                        TOTAL_VENTA,
                        ID_USUARIO
                    FROM VENTA
                    WHERE DATE(FECHA_REGISTRO_VENTA) = CURDATE()
                    ORDER BY FECHA_REGISTRO_VENTA DESC, ID_VENTA DESC";

            $stmt = $this->db->query($sql);
            $ventas = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];

            // (Opcional) Formatear fecha si quieres un formato específico:
            // foreach ($ventas as &$v) {
            //     if (!empty($v['FECHA_REGISTRO_VENTA'])) {
            //         $v['FECHA_REGISTRO_VENTA'] = date('Y-m-d H:i:s', strtotime($v['FECHA_REGISTRO_VENTA']));
            //     }
            // }

            return $ventas;

        } catch (PDOException $e) {
            return [];
        }
    }
}
