<?php
    require __DIR__ . '/../../../vendor/autoload.php';

    // Conexión procedural
    $pathConn = __DIR__ . '/../../database/Conexion_SQL_Server.php';
    if (!file_exists($pathConn)) die("No se encontró el archivo de conexión: $pathConn");
    require_once $pathConn;
    if (!isset($conn) || $conn === false) die('No se pudo obtener la conexión $conn');

    use PhpOffice\PhpSpreadsheet\Spreadsheet;
    use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
    use PhpOffice\PhpSpreadsheet\Style\Border;
    use PhpOffice\PhpSpreadsheet\Style\Fill;
    use PhpOffice\PhpSpreadsheet\Style\Alignment;
    use PhpOffice\PhpSpreadsheet\Cell\Coordinate;

    date_default_timezone_set('America/Lima');

    // Convierte 1->A, 27->AA
    function colLetter($n) { return Coordinate::stringFromColumnIndex($n); }
    // Rango A1 a última celda (colCount, row)
    function rangeA1($colCount, $row) { return 'A1:' . colLetter($colCount) . $row; }

    // ========= Exportar una hoja =========
    function exportarHoja(Spreadsheet $spreadsheet, $conn, string $tituloHoja, string $sql, array $colsMoneda = []) {
        $hoja = $spreadsheet->createSheet();
        $hoja->setTitle($tituloHoja);

        $stmt = sqlsrv_query($conn, $sql);
        if ($stmt === false) die("Error SQL ($tituloHoja): " . print_r(sqlsrv_errors(), true));

        // Encabezados
        $meta = sqlsrv_field_metadata($stmt);
        $colCount = count($meta);
        for ($c = 1; $c <= $colCount; $c++) {
            $cell = colLetter($c) . '1';
            $hoja->setCellValue($cell, $meta[$c-1]['Name']);
        }

        // Datos
        $rowNum = 2;
        while ($row = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
            $c = 1;
            foreach ($row as $v) {
                if ($v instanceof DateTimeInterface) $v = $v->format('Y-m-d H:i:s');
                $cell = colLetter($c) . $rowNum;
                $hoja->setCellValue($cell, $v);
                $c++;
            }
            $rowNum++;
        }
        sqlsrv_free_stmt($stmt);

        if ($colCount === 0) return;

        $ultFila = $rowNum - 1;
        $ultColLetter = colLetter($colCount);

        // Estilo encabezado (centrado)
        $hoja->getStyle("A1:{$ultColLetter}1")->applyFromArray([
            'font'      => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']],
            'fill'      => ['fillType' => Fill::FILL_SOLID, 'color' => ['rgb' => '4F81BD']],
            'borders'   => ['allBorders' => ['borderStyle' => Border::BORDER_THIN]],
            'alignment' => ['horizontal' => Alignment::HORIZONTAL_CENTER, 'vertical' => Alignment::VERTICAL_CENTER],
        ]);

        // Bordes a todo
        $hoja->getStyle(rangeA1($colCount, $ultFila))->applyFromArray([
            'borders' => ['allBorders' => ['borderStyle' => Border::BORDER_THIN]]
        ]);

        // Centrar datos (A2:última)
        if ($ultFila >= 2) {
            $hoja->getStyle("A2:{$ultColLetter}{$ultFila}")
                ->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
            $hoja->getStyle("A2:{$ultColLetter}{$ultFila}")
                ->getAlignment()->setVertical(Alignment::VERTICAL_CENTER);
        }

        // Autofiltro y freeze encabezado
        $hoja->setAutoFilter("A1:{$ultColLetter}{$ultFila}");
        $hoja->freezePane('A2');

        // Ancho auto
        for ($c = 1; $c <= $colCount; $c++) {
            $hoja->getColumnDimension(colLetter($c))->setAutoSize(true);
        }

        // Formato moneda
        if (!empty($colsMoneda) && $ultFila >= 2) {
            for ($i = 0; $i < $colCount; $i++) {
                $nombreCampo = $meta[$i]['Name'];
                if (in_array($nombreCampo, $colsMoneda, true)) {
                    $colL = colLetter($i + 1);
                    $hoja->getStyle("{$colL}2:{$colL}{$ultFila}")
                        ->getNumberFormat()->setFormatCode('#,##0.00');
                }
            }
        }
    }

    // ========= Crear libro =========
    $excel = new Spreadsheet();
    $excel->removeSheetByIndex(0);

    // Hoja 1: APERTURA_CAJA
    $sqlApertura = "
        SELECT FECHA_APERTURA, ID_APERTURA, MONTO_APERTURA, ID_USUARIO
        FROM APERTURA_CAJA
        ORDER BY ID_APERTURA DESC
    ";
    exportarHoja($excel, $conn, 'Apertura_Caja', $sqlApertura, ['MONTO_APERTURA']);

    // Hoja 2: VENTA (orden descendente por ID_VENTA)
    $sqlVenta = "
        SELECT FECHA_REGISTRO_VENTA, ID_VENTA, ID_PRODUCTO, CANTIDAD,
            MONTO_PRODUCTO, DESCUENTO_PRODUCTO, METODO_PAGO,
            TOTAL_VENTA, ID_USUARIO
        FROM VENTA
        ORDER BY ID_VENTA DESC
    ";
    exportarHoja($excel, $conn, 'Registro_Venta', $sqlVenta, ['MONTO_PRODUCTO','DESCUENTO_PRODUCTO','TOTAL_VENTA']);

    // Hoja 3: REPORTE_VENTA
    $sqlReporte = "
        SELECT FECHA_REPORT_VENTA, ID_REPORTE, MONTO_APERTURA,
            TOTAL_EFECTIVO, TOTAL_YAPE_PLIN, TOTAL_TARJETA,
            TOTAL_DESCUENTO, TOTAL_REPORTE, ID_USUARIO
        FROM REPORTE_VENTA
        ORDER BY ID_REPORTE DESC
    ";
    exportarHoja($excel, $conn, 'Reporte_Venta', $sqlReporte,
        ['MONTO_APERTURA','TOTAL_EFECTIVO','TOTAL_YAPE_PLIN','TOTAL_TARJETA','TOTAL_DESCUENTO','TOTAL_REPORTE']
    );

    // ======= Hoja 4: CIERRE_CAJA (NUEVA) =======
    $sqlCierre = "
        SELECT
            FECHA_CIERRE,
            ID_CIERRE,
            MONTO_APERTURA_DIA,
            TOTAL_EFECTIVO_DIA,
            TOTAL_YAPE_PLIN_DIA,
            TOTAL_TARJETA_DIA,
            TOTAL_REPORTE_DIA,
            TOTAL_RETIROS_DIA,
            TOTAL_INGRESOS_DIA,
            EFECTIVO_SISTEMA_FINAL,
            CONTEO_FISICO_CAJA,
            DINERO_SOBRANTE,
            DINERO_FALTANTE,
            OBSERVACION_CIERRE,
            ID_USUARIO
        FROM CIERRE_CAJA
        ORDER BY ID_CIERRE DESC
    ";
    exportarHoja(
        $excel, $conn, 'Cierre_Caja', $sqlCierre,
        [
            'MONTO_APERTURA_DIA',
            'TOTAL_EFECTIVO_DIA',
            'TOTAL_YAPE_PLIN_DIA',
            'TOTAL_TARJETA_DIA',
            'TOTAL_REPORTE_DIA',
            'TOTAL_RETIROS_DIA',
            'TOTAL_INGRESOS_DIA',
            'EFECTIVO_SISTEMA_FINAL',
            'CONTEO_FISICO_CAJA',
            'DINERO_SOBRANTE',
            'DINERO_FALTANTE'
        ]
    );

    $excel->setActiveSheetIndex(0);

    // Guardar en E:\
    $ruta = 'E:\\Archivo_Tulinda.xlsx';
    $writer = new Xlsx($excel);
    try {
        $writer->save($ruta);
        echo "Archivo exportado correctamente";
    } catch (Throwable $e) {
        http_response_code(500);
        echo "ERROR: " . $e->getMessage();
    }
?>