<?php
session_start();

// [1] Verifica sesión
if (!isset($_SESSION['id_usuario'])) {
  header('Location: Login_Tulinda.html?error=Sesion expirada'); 
  exit;
}

// [2] Permitir solo VENDEDORA o ADMIN
if (!in_array($_SESSION['rol'] ?? '', ['VENDEDORA', 'ADMIN'], true)) {
  header('Location: home.php?error=sin_permiso'); 
  exit;
}

// [3] Variables de sesión
$idUsuario       = $_SESSION['id_usuario'];
$rol             = $_SESSION['rol'] ?? '';
$estado          = $_SESSION['estado'] ?? '';
$nombreVendedora = $_SESSION['nombre_vendedora'] ?? $idUsuario;

// [4] Conexión a base de datos (PDO MySQL)
require_once __DIR__ . '/../backend/database/Conexion_DataBase.php';

// [5] Cargar productos activos (PDO)
$productosActivos = [];
try {
  $sqlProd = "SELECT ID_PRODUCTO, TIPO_PRODUCTO, PRECIO_PRODUCTO
              FROM PRODUCTO
              WHERE ESTADO_PRODUCTO = 'ACTIVO'
              ORDER BY TIPO_PRODUCTO ASC";
  $stmt = $pdo->query($sqlProd);
  $productosActivos = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
} catch (Throwable $e) {
  $productosActivos = [];
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8" />
  <title>Tulinda • Apertura y Venta</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />

  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
  <link rel="stylesheet" href="../css/Estilos_RegistroVenta.css">
</head>
<body>

<header class="header-tulinda px-3 py-2 shadow-sm">
  <div class="row align-items-center g-2 w-100">
    <!-- IZQUIERDA: usuario -->
    <div class="col-12 col-md-4 d-flex align-items-center gap-2 flex-wrap user-block">
      <div class="avatar"><i class="bi bi-person-fill"></i></div>
      <div class="lh-1 text-truncate user-meta">
        <strong class="text-truncate d-block">
          <?= htmlspecialchars($nombreVendedora, ENT_QUOTES, 'UTF-8'); ?>
        </strong>
        <small class="text-muted text-truncate d-block">
          <?= htmlspecialchars($rol, ENT_QUOTES, 'UTF-8'); ?>
        </small>
      </div>

      <?php if ($rol === 'ADMIN'): ?>
        <a href="./app_Inicio.php"
           class="btn btn-warning text-nowrap d-none d-md-inline-flex align-items-center ms-auto">
          <i class="bi bi-speedometer2 me-1"></i> Ir a Interfaz Administrador
        </a>
      <?php endif; ?>
    </div>

    <!-- CENTRO: chip -->
    <div class="col-12 col-md-4 d-flex justify-content-center">
      <a href="./View_Vendedora.php" class="chip-link text-center">
        <i class="bi bi-bar-chart-line me-1"></i> Ver Registros de Venta
      </a>
    </div>

    <!-- DERECHA: acciones -->
    <div class="col-12 col-md-4">
      <div class="header-actions d-flex flex-wrap align-items-center justify-content-start justify-content-md-end gap-2">
        <form action="../backend/procesos/Generar_Reporte_Venta.php" method="post" class="m-0">
          <button type="submit" class="btn btn-outline-success text-nowrap">
            <i class="bi bi-receipt-cutoff"></i> Generar Reporte del Día
          </button>
        </form>

        <form action="../backend/procesos/seguridad/logout.php" method="post" class="m-0"
              onsubmit="return confirm('¿Deseas cerrar sesión?');">
          <button type="submit" class="btn btn-rose-outline text-nowrap">
            <i class="bi bi-box-arrow-right"></i> Cerrar sesión
          </button>
        </form>
      </div>

      <?php if ($rol === 'ADMIN'): ?>
        <a href="./View_Admin.php"
           class="btn btn-warning text-nowrap d-inline-flex d-md-none align-items-center mt-2 w-100">
          <i class="bi bi-speedometer2 me-1"></i> Ir a Interfaz Administrador
        </a>
      <?php endif; ?>
    </div>
  </div>
</header>

<div class="container container-tulinda mt-4">

  <?php if(isset($_GET['ok']) || isset($_GET['err']) || isset($_GET['msg']) || isset($_GET['error'])): ?>
    <div id="msgAlert" class="alert alert-dismissible fade show alert-fixed
         <?= (isset($_GET['ok']) || isset($_GET['msg'])) ? 'alert-success' : 'alert-danger' ?>">
      <?= isset($_GET['ok'])   ? htmlspecialchars($_GET['ok'])   :
         (isset($_GET['msg']) ? htmlspecialchars($_GET['msg']) :
         (isset($_GET['err']) ? htmlspecialchars($_GET['err']) : htmlspecialchars($_GET['error']))) ?>
      <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
  <?php endif; ?>

  <!-- DOS BLOQUES -->
  <div class="row g-4 mb-5">
    <!-- COLUMNA 1: Apertura -->
    <div class="col-lg-5 col-md-6">
      <section class="panel-card p-3 p-md-4 h-100">
        <h2 class="h-title mb-2">Apertura de Caja</h2>
        <div class="rule-rose"></div>

        <form action="../backend/procesos/Guardar_Apertura.php" method="post" autocomplete="off">
          <div class="mb-3">
            <label for="monto_apertura" class="form-label">Monto inicial en caja</label>
            <input type="number" step="0.01" min="0" class="form-control" id="monto_apertura"
                   name="monto_apertura" required placeholder="0.00">
          </div>
          <button type="submit" class="btn btn-rose">Registrar Apertura</button>
        </form>
      </section>
    </div>

    <!-- COLUMNA 2: Venta -->
    <div class="col-lg-7 col-md-6">
      <section class="panel-card p-3 p-md-4 h-100">
        <h3 class="h-title mb-2">Registrar Venta</h3>
        <div class="rule-rose"></div>

        <form action="../backend/procesos/Guardar_Venta.php" method="post" autocomplete="off">
          <div class="mb-3">
            <label for="id_producto" class="form-label">Producto</label>
            <select class="form-select" id="id_producto" name="id_producto" required>
              <option value="" selected disabled>Seleccione un producto…</option>
              <?php foreach($productosActivos as $p): ?>
                <option value="<?= htmlspecialchars($p['ID_PRODUCTO']) ?>"
                        data-precio="<?= (float)$p['PRECIO_PRODUCTO'] ?>">
                  <?= htmlspecialchars($p['ID_PRODUCTO'] . ' — ' . $p['TIPO_PRODUCTO']) ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="row">
            <div class="col-md-6 mb-3">
              <label for="cantidad" class="form-label">Cantidad</label>
              <input type="number" name="cantidad" id="cantidad" class="form-control" required min="1"
                     step="1" placeholder="1">
            </div>

            <div class="col-md-6 mb-3">
              <label for="monto_producto" class="form-label">Precio unitario</label>
              <input type="number" step="0.01" name="monto_producto" id="monto_producto"
                     class="form-control" required min="0" placeholder="0.00" disabled>
            </div>
          </div>

          <div class="row">
            <div class="col-md-6 mb-3">
              <label for="descuento_producto" class="form-label">Descuento</label>
              <input type="number" step="0.01" name="descuento_producto" id="descuento_producto"
                     class="form-control" min="0" placeholder="0.00">
            </div>

            <div class="col-md-6 mb-3">
              <label for="metodo_pago" class="form-label">Método de pago</label>
              <select name="metodo_pago" id="metodo_pago" class="form-select" required>
                <option value="">Seleccione…</option>
                <option value="EFECTIVO">Efectivo</option>
                <option value="YAPE_PLIN">Yape / Plin</option>
                <option value="TARJETA">Tarjeta</option>
              </select>
            </div>
          </div>

          <!-- Total estimado -->
          <div class="total-preview" id="totalBox" aria-live="polite">
            <span>Total estimado</span>
            <span id="totalPreview">S/ 0.00</span>
          </div>

          <div class="mt-3">
            <button type="submit" class="btn btn-rose" id="btnRegistrarVenta">Registrar Venta</button>
          </div>
        </form>
      </section>
    </div>
  </div>

  <p class="text-center mb-4 footer-mini">Tulinda – Joyería Artesanal</p>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', () => {
  const selectProducto  = document.getElementById('id_producto');
  const inputPrecio     = document.getElementById('monto_producto');
  const inputCantidad   = document.getElementById('cantidad');
  const inputDescuento  = document.getElementById('descuento_producto');
  const totalPreview    = document.getElementById('totalPreview');
  const btnVenta        = document.getElementById('btnRegistrarVenta');

  // Autollenado del precio desde data-precio
  selectProducto.addEventListener('change', () => {
    const opcion = selectProducto.options[selectProducto.selectedIndex];
    const precio = opcion.getAttribute('data-precio');
    inputPrecio.value = precio ? parseFloat(precio).toFixed(2) : '';
    actualizarTotal();
  });

  [inputPrecio, inputCantidad, inputDescuento].forEach(el => {
    el.addEventListener('input', actualizarTotal);
  });

  function actualizarTotal() {
    const cant = parseFloat(inputCantidad.value) || 0;
    const p    = parseFloat(inputPrecio.value)  || 0;
    const d    = parseFloat(inputDescuento.value) || 0;

    const total = (cant * p) - d;
    totalPreview.textContent = 'S/ ' + (isFinite(total) ? total.toFixed(2) : '0.00');

    const invalido = (cant < 1) || (p < 0) || (d < 0) || (total < 0);
    btnVenta.disabled = invalido;
  }

  actualizarTotal();
});
</script>
</body>
</html>
