<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\CierreCaja;
use App\Models\AperturaCaja;
use App\Models\Venta;
use App\Models\ReporteVenta;
use Carbon\Carbon;

class CierreCajaController extends Controller
{
    private function ensureAllowed(array $roles)
    {
        if (!in_array(session('usuario.ROL'), $roles)) abort(403);
    }

    /**
     * ✅ Devuelve el reporte de HOY si existe, si no retorna null
     */
    private function reporteHoy()
    {
        $hoy = Carbon::today();
        return ReporteVenta::whereDate('FECHA_REPORTE', $hoy)->first();
    }

    public function index()
    {
        $this->ensureAllowed(['ADMIN', 'VENDEDOR']);

        $hoy = Carbon::today();

        // ✅ SOLO CIERRES DE HOY (ya no historial)
        $cierres = CierreCaja::query()
            ->whereDate('FECHA_CIERRE_CAJA', $hoy)
            ->orderByDesc('FECHA_CIERRE_CAJA')
            ->paginate(10);

        // Datos del día (para tarjetas)
        $aperturaHoy = AperturaCaja::whereDate('FECHA_APERTURA', $hoy)->first();
        $cierreHoy   = CierreCaja::whereDate('FECHA_CIERRE_CAJA', $hoy)->first();
        $reporteHoy  = $this->reporteHoy();

        $totalHoy = (float) Venta::whereDate('FECHA_REGISTRO', $hoy)->sum('TOTAL_VENTA');
        $efectivo = (float) Venta::whereDate('FECHA_REGISTRO', $hoy)->where('METODO_PAGO', 'EFECTIVO')->sum('TOTAL_VENTA');
        $yape     = (float) Venta::whereDate('FECHA_REGISTRO', $hoy)->where('METODO_PAGO', 'YAPE_PLIN')->sum('TOTAL_VENTA');
        $tarjeta  = (float) Venta::whereDate('FECHA_REGISTRO', $hoy)->where('METODO_PAGO', 'TARJETA')->sum('TOTAL_VENTA');

        $montoApertura = $aperturaHoy ? (float) $aperturaHoy->MONTO_APERTURA : 0;

        // Lo que debería haber en caja (efectivo)
        $efectivoSistema = $montoApertura + $efectivo;

        return view('cierre.index', compact(
            'cierres',
            'aperturaHoy',
            'cierreHoy',
            'reporteHoy',
            'totalHoy',
            'efectivo',
            'yape',
            'tarjeta',
            'montoApertura',
            'efectivoSistema'
        ));
    }

    public function create()
    {
        $this->ensureAllowed(['ADMIN', 'VENDEDOR']);

        $hoy = Carbon::today();

        // ✅ 1) Debe existir apertura
        $aperturaHoy = AperturaCaja::whereDate('FECHA_APERTURA', $hoy)->first();
        if (!$aperturaHoy) {
            return redirect()->route('apertura.create')
                ->with('error', 'No hay apertura hoy. Primero debes abrir caja.');
        }

        // ✅ 2) NO debe existir cierre hoy
        if (CierreCaja::whereDate('FECHA_CIERRE_CAJA', $hoy)->exists()) {
            return redirect()->route('cierre.index')->with('error', 'Ya existe un cierre hoy.');
        }

        // ✅ 3) Debe existir REPORTE hoy
        $reporteHoy = $this->reporteHoy();
        if (!$reporteHoy) {
            return redirect()->route('reporte.index')
                ->with('error', '⚠️ Primero debes generar el REPORTE del día antes de cerrar caja.');
        }

        // cálculos
        $efectivo = (float) Venta::whereDate('FECHA_REGISTRO', $hoy)
            ->where('METODO_PAGO', 'EFECTIVO')
            ->sum('TOTAL_VENTA');

        $montoApertura = (float) $aperturaHoy->MONTO_APERTURA;
        $efectivoSistema = $montoApertura + $efectivo;

        return view('cierre.create', compact(
            'aperturaHoy',
            'reporteHoy',
            'montoApertura',
            'efectivo',
            'efectivoSistema'
        ));
    }

    public function store(Request $request)
    {
        $this->ensureAllowed(['ADMIN', 'VENDEDOR']);

        $request->validate([
            'EFECTIVO_FISICO' => 'required|numeric|min:0'
        ]);

        $hoy = Carbon::today();

        // ✅ 1) Debe existir apertura
        $aperturaHoy = AperturaCaja::whereDate('FECHA_APERTURA', $hoy)->first();
        if (!$aperturaHoy) {
            return redirect()->route('apertura.create')
                ->with('error', 'No hay apertura hoy. Primero debes abrir caja.');
        }

        // ✅ 2) NO debe existir cierre hoy
        if (CierreCaja::whereDate('FECHA_CIERRE_CAJA', $hoy)->exists()) {
            return redirect()->route('cierre.index')->with('error', 'Ya existe un cierre hoy.');
        }

        // ✅ 3) Debe existir REPORTE hoy
        $reporteHoy = $this->reporteHoy();
        if (!$reporteHoy) {
            return redirect()->route('reporte.index')
                ->with('error', '⚠️ No se puede cerrar caja sin generar el REPORTE del día.');
        }

        // cálculos
        $efectivo = (float) Venta::whereDate('FECHA_REGISTRO', $hoy)
            ->where('METODO_PAGO', 'EFECTIVO')
            ->sum('TOTAL_VENTA');

        $montoApertura   = (float) $aperturaHoy->MONTO_APERTURA;
        $efectivoSistema = $montoApertura + $efectivo;

        $fisico = (float) $request->EFECTIVO_FISICO;

        $sobrante = 0;
        $faltante = 0;
        $obs = 'CIERRE CORRECTO';

        if ($fisico > $efectivoSistema) {
            $sobrante = $fisico - $efectivoSistema;
            $obs = 'SOBRA DINERO: S/ ' . number_format($sobrante, 2);
        } elseif ($fisico < $efectivoSistema) {
            $faltante = $efectivoSistema - $fisico;
            $obs = 'FALTA DINERO: S/ ' . number_format($faltante, 2);
        }

        CierreCaja::create([
            'FECHA_CIERRE_CAJA'  => now(),
            'ID_APERTURA'        => $aperturaHoy->ID_APERTURA,
            'ID_REPORTE'         => $reporteHoy->ID_REPORTE,

            'MONTO_APERTURA'     => $montoApertura,
            'EFECTIVO_VENTAS'    => $efectivo,
            'EFECTIVO_SISTEMA'   => $efectivoSistema,
            'EFECTIVO_FISICO'    => $fisico,
            'DINERO_SOBRANTE'    => $sobrante,
            'DINERO_FALTANTE'    => $faltante,
            'OBSERVACION_CIERRE' => $obs,
            'ESTADO_CIERRE'      => 'CERRADO',
            'ID_EMPLEADO'        => session('usuario.ID_EMPLEADO') ?? 'N/A',
        ]);

        return redirect()->route('cierre.index')->with('ok', 'Cierre registrado.');
    }
}
