<?php

namespace App\Http\Controllers;

use App\Models\AperturaCaja;
use App\Models\CierreCaja;
use App\Models\ReporteVenta;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index()
    {
        $u = session('usuario', []);
        $rol = $u['ROL'] ?? '';
        $user = $u['ID_USUARIO'] ?? 'usuario';

        $aperturaHoy = AperturaCaja::whereDate('FECHA_APERTURA', now())->first();
        $cierreHoy   = CierreCaja::whereDate('FECHA_CIERRE_CAJA', now())->first();
        $reporteHoy  = ReporteVenta::whereDate('FECHA_REPORTE', now())->first();

        // ✅ Años disponibles desde reporte_ventas
        $anios = ReporteVenta::query()
            ->selectRaw('YEAR(FECHA_REPORTE) as anio')
            ->distinct()
            ->orderByDesc('anio')
            ->pluck('anio');

        // ✅ Productos: tu tabla NO tiene NOMBRE_PRODUCTO
        // Usamos TIPO_PRODUCTO como texto visible
        $productos = DB::table('productos')
            ->select('ID_PRODUCTO', 'TIPO_PRODUCTO', 'PRECIO_PRODUCTO', 'ESTADO_PRODUCTO')
            ->orderBy('TIPO_PRODUCTO')
            ->orderBy('ID_PRODUCTO')
            ->get();

        // Defaults
        $anioActual = (int) Carbon::now()->year;
        $anioPasado = $anioActual - 1;

        $anioA = $anios->first() ?? $anioActual;
        $anioB = $anios->skip(1)->first() ?? $anioPasado;

        return view('dashboard.index', compact(
            'rol',
            'user',
            'aperturaHoy',
            'cierreHoy',
            'reporteHoy',
            'anios',
            'productos',
            'anioA',
            'anioB'
        ));
    }

    /**
     * ✅ JSON para el gráfico:
     * - Compara Año A vs Año B
     * - Permite elegir métrica: total/efectivo/yape/tarjeta/descuentos
     * - PROTEGIDO: SOLO ADMIN
     */
    public function data(Request $request)
    {
        // ✅ seguridad por rol
        $u = session('usuario', []);
        $rol = $u['ROL'] ?? '';
        if ($rol !== 'ADMIN') {
            return response()->json(['message' => 'No autorizado'], 403);
        }

        $anioA = (int) $request->get('anioA');
        $anioB = (int) $request->get('anioB');

        // ✅ NUEVO: métrica seleccionable (por tipo de pago)
        $metric = $request->get('metric', 'total'); // default total

        $map = [
            'total'      => 'TOTAL_REPORTE',
            'efectivo'   => 'EFECTIVO_REPORTE',
            'yape'       => 'YAPE_PLIN_REPORTE',
            'tarjeta'    => 'TARJETA_REPORTE',
            'descuentos' => 'TOTAL_DESCUENTOS',
        ];

        $col = $map[$metric] ?? 'TOTAL_REPORTE';

        $labels = ['Ene', 'Feb', 'Mar', 'Abr', 'May', 'Jun', 'Jul', 'Ago', 'Sep', 'Oct', 'Nov', 'Dic'];

        $getSerie = function ($anio) use ($col) {
            $rows = ReporteVenta::query()
                ->selectRaw("MONTH(FECHA_REPORTE) as mes, SUM($col) as total")
                ->whereYear('FECHA_REPORTE', $anio)
                ->groupBy('mes')
                ->orderBy('mes')
                ->pluck('total', 'mes');

            $serie = [];
            for ($m = 1; $m <= 12; $m++) {
                $serie[] = (float)($rows[$m] ?? 0);
            }
            return $serie;
        };

        return response()->json([
            'labels' => $labels,
            'serieA' => $getSerie($anioA),
            'serieB' => $getSerie($anioB),
            'metric' => $metric,
            'col'    => $col,
            'tipo'   => 'reporte_ventas'
        ]);
    }
}
