<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\ReporteVenta;
use App\Models\Venta;
use Carbon\Carbon;

class ReporteVentaController extends Controller
{
    private function ensureAllowed(array $roles)
    {
        if (!in_array(session('usuario.ROL'), $roles)) abort(403);
    }

    public function index()
    {
        $this->ensureAllowed(['ADMIN', 'VENDEDOR']);

        $rol = session('usuario.ROL') ?? '';
        $hoy = Carbon::today();

        // ✅ VENDEDOR: solo HOY
        if ($rol === 'VENDEDOR') {
            $reporteHoy = ReporteVenta::whereDate('FECHA_REPORTE', $hoy)
                ->orderBy('FECHA_REPORTE', 'desc')
                ->first();

            // Enviamos una "lista" para reusar la vista (puede ser collection vacía)
            $reportes = collect($reporteHoy ? [$reporteHoy] : []);
            $soloHoy = true;

            return view('reporte.index', compact('reportes', 'reporteHoy', 'soloHoy'));
        }

        // ✅ ADMIN: historial completo (paginado)
        $reportes = ReporteVenta::orderBy('FECHA_REPORTE', 'desc')
            ->paginate(10);

        $reporteHoy = ReporteVenta::whereDate('FECHA_REPORTE', $hoy)->exists();
        $soloHoy = false;

        return view('reporte.index', compact('reportes', 'reporteHoy', 'soloHoy'));
    }

    public function generarHoy(Request $request)
    {
        $this->ensureAllowed(['ADMIN', 'VENDEDOR']);

        // 1 por día
        if (ReporteVenta::whereDate('FECHA_REPORTE', now())->exists()) {
            return redirect()->route('reporte.index')
                ->with('error', 'Ya existe un reporte generado hoy.');
        }

        // Sumatorias por método (hoy)
        $efectivo = (float) Venta::whereDate('FECHA_REGISTRO', now())
            ->where('METODO_PAGO', 'EFECTIVO')
            ->sum('TOTAL_VENTA');

        $yape = (float) Venta::whereDate('FECHA_REGISTRO', now())
            ->where('METODO_PAGO', 'YAPE_PLIN')
            ->sum('TOTAL_VENTA');

        $tarjeta = (float) Venta::whereDate('FECHA_REGISTRO', now())
            ->where('METODO_PAGO', 'TARJETA')
            ->sum('TOTAL_VENTA');

        $descuentos = (float) Venta::whereDate('FECHA_REGISTRO', now())->sum('DESCUENTO_PRODUCTO');
        $total = (float) Venta::whereDate('FECHA_REGISTRO', now())->sum('TOTAL_VENTA');

        ReporteVenta::create([
            'FECHA_REPORTE'        => now(),
            'EFECTIVO_REPORTE'     => $efectivo,
            'YAPE_PLIN_REPORTE'    => $yape,
            'TARJETA_REPORTE'      => $tarjeta,
            'TOTAL_DESCUENTOS'     => $descuentos,
            'TOTAL_REPORTE'        => $total,
            'ID_EMPLEADO'          => session('usuario.ID_EMPLEADO') ?? 'N/A',
        ]);

        return redirect()->route('reporte.index')->with('ok', 'Reporte de hoy generado.');
    }
}
