<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Models\Usuario;
use App\Models\Empleado;


class UsuarioController extends Controller
{
    private function ensureAdmin()
    {
        $rol = session('usuario.ROL');
        if ($rol !== 'ADMIN') {
            abort(403, 'No tienes permiso.');
        }
    }

    public function index()
    {
        $this->ensureAdmin();

        $usuarios = Usuario::with('empleado')->orderBy('created_at', 'desc')->paginate(10);
        return view('usuarios.index', compact('usuarios'));
    }

    public function create()
    {
        $this->ensureAdmin();

        $empleados = Empleado::orderBy('NOMBRE_EMPLEADO')->get();
        return view('usuarios.create', compact('empleados'));
    }

    public function store(Request $request)
    {
        $this->ensureAdmin();

        $data = $request->validate([
            'ID_USUARIO' => 'required|string|max:20|unique:usuarios,ID_USUARIO',
            'ID_EMPLEADO' => 'required|exists:empleados,ID_EMPLEADO',
            'CONTRASENA' => 'required|string|min:4|max:50',
            'ROL' => 'required|in:ADMIN,VENDEDOR',
            'ESTADO_USUARIO' => 'required|in:ACTIVO,INACTIVO',
        ]);

        $data['CONTRASENA'] = Hash::make($data['CONTRASENA']);

        Usuario::create($data);
        return redirect()->route('usuarios.index')->with('ok', 'Usuario registrado.');
    }

    public function edit(Usuario $usuario)
    {
        $this->ensureAdmin();

        $empleados = Empleado::orderBy('NOMBRE_EMPLEADO')->get();
        return view('usuarios.edit', compact('usuario', 'empleados'));
    }

    public function update(Request $request, Usuario $usuario)
    {
        $this->ensureAdmin();

        $data = $request->validate([
            'ID_EMPLEADO' => 'required|exists:empleados,ID_EMPLEADO',
            'CONTRASENA' => 'nullable|string|min:4|max:50',
            'ROL' => 'required|in:ADMIN,VENDEDOR',
            'ESTADO_USUARIO' => 'required|in:ACTIVO,INACTIVO',
        ]);

        if (!empty($data['CONTRASENA'])) {
            $data['CONTRASENA'] = Hash::make($data['CONTRASENA']);
        } else {
            unset($data['CONTRASENA']);
        }

        $usuario->update($data);
        return redirect()->route('usuarios.index')->with('ok', 'Usuario actualizado.');
    }

    public function destroy(Usuario $usuario)
    {
        $this->ensureAdmin();

        $usuario->delete();
        return redirect()->route('usuarios.index')->with('ok', 'Usuario eliminado.');
    }
}
