<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Venta;
use App\Models\Producto;
use Carbon\Carbon;

// ✅ MODELOS REALES DE TU PROYECTO (según tu screenshot)
use App\Models\CierreCaja;
use App\Models\ReporteVenta;

class VentaController extends Controller
{
    private function ensureAllowed(array $roles)
    {
        if (!in_array(session('usuario.ROL'), $roles)) abort(403);
    }

    /**
     * ✅ BLOQUEO: si HOY ya existe CIERRE o REPORTE, no se puede vender más.
     */
    private function diaBloqueado(): bool
    {
        $hoy = Carbon::today();

        $cierreHoy  = CierreCaja::whereDate('FECHA_CIERRE_CAJA', $hoy)->exists();
        $reporteHoy = ReporteVenta::whereDate('FECHA_REPORTE', $hoy)->exists();

        return $cierreHoy || $reporteHoy;
    }

    public function index(Request $request)
    {
        $this->ensureAllowed(['ADMIN', 'VENDEDOR']);

        $fecha = $request->input('fecha', now()->toDateString());
        $metodo = $request->input('metodo');
        $q = $request->input('q');

        $query = Venta::query()->whereDate('FECHA_REGISTRO', $fecha);

        if ($metodo) $query->where('METODO_PAGO', $metodo);
        if ($q) $query->where('ID_PRODUCTO', 'like', "%$q%");

        $ventas = $query->orderBy('FECHA_REGISTRO', 'desc')->paginate(10)->withQueryString();

        $baseTotales = Venta::whereDate('FECHA_REGISTRO', $fecha);
        $totales = [
            'total'    => (float) $baseTotales->sum('TOTAL_VENTA'),
            'efectivo' => (float) Venta::whereDate('FECHA_REGISTRO', $fecha)->where('METODO_PAGO', 'EFECTIVO')->sum('TOTAL_VENTA'),
            'yape'     => (float) Venta::whereDate('FECHA_REGISTRO', $fecha)->where('METODO_PAGO', 'YAPE_PLIN')->sum('TOTAL_VENTA'),
            'tarjeta'  => (float) Venta::whereDate('FECHA_REGISTRO', $fecha)->where('METODO_PAGO', 'TARJETA')->sum('TOTAL_VENTA'),
        ];

        // ✅ Para bloquear el botón en la vista
        $diaBloqueado = $this->diaBloqueado();

        return view('ventas.index', compact('ventas', 'totales', 'diaBloqueado'));
    }

    public function create()
    {
        $this->ensureAllowed(['ADMIN', 'VENDEDOR']);

        // ✅ Bloqueo real
        if ($this->diaBloqueado()) {
            return redirect()
                ->route('ventas.index')
                ->with('error', 'Día cerrado: ya se generó el reporte o cierre de hoy. Ventas habilitadas mañana.');
        }

        $productos = Producto::where('ESTADO_PRODUCTO', 'ACTIVO')->get();
        return view('ventas.create', compact('productos'));
    }

    public function store(Request $request)
    {
        $this->ensureAllowed(['ADMIN', 'VENDEDOR']);

        // ✅ Bloqueo real (seguridad extra)
        if ($this->diaBloqueado()) {
            return redirect()
                ->route('ventas.index')
                ->with('error', 'Bloqueado: ya se generó el reporte o cierre de hoy. Ventas habilitadas mañana.');
        }

        $data = $request->validate([
            'ID_PRODUCTO' => 'required|exists:productos,ID_PRODUCTO',
            'CANTIDAD' => 'required|integer|min:1',
            'DESCUENTO_PRODUCTO' => 'nullable|numeric|min:0',
            'METODO_PAGO' => 'required|in:EFECTIVO,YAPE_PLIN,TARJETA',
        ]);

        $producto = Producto::where('ID_PRODUCTO', $data['ID_PRODUCTO'])->firstOrFail();

        $subtotal = $producto->PRECIO_PRODUCTO * $data['CANTIDAD'];
        $descuento = $data['DESCUENTO_PRODUCTO'] ?? 0;

        if ($descuento > $subtotal) $descuento = $subtotal;

        $total = max(0, $subtotal - $descuento);

        Venta::create([
            'FECHA_REGISTRO' => now(),
            'ID_PRODUCTO' => $data['ID_PRODUCTO'],
            'CANTIDAD' => $data['CANTIDAD'],
            'MONTO_PRODUCTO' => $subtotal,
            'DESCUENTO_PRODUCTO' => $descuento,
            'METODO_PAGO' => $data['METODO_PAGO'],
            'TOTAL_VENTA' => $total,
            'ID_EMPLEADO' => session('usuario.ID_EMPLEADO'),
        ]);

        return redirect()->route('ventas.index')->with('ok', 'Venta registrada.');
    }
}
