@extends('layouts.app')
@section('title', 'Reporte de Ventas')

@push('styles')
    @vite(['resources/css/reporte.css'])
@endpush

@section('content')
    @php
        $rol = session('usuario.ROL') ?? 'INVITADO';

        // ✅ Si es VENDEDOR: $reportes puede venir como Collection (no paginate)
        // ✅ Si es ADMIN: $reportes viene como paginator

        // Sacamos "hoy" como el primer item de la lista que llega
        // - ADMIN: first() será el último reporte (por order desc)
        // - VENDEDOR: first() será el reporte de HOY (si existe)
        $hoy = $reportes->first();

        $ef = $hoy->EFECTIVO_REPORTE ?? 0;
        $yp = $hoy->YAPE_PLIN_REPORTE ?? 0;
        $tj = $hoy->TARJETA_REPORTE ?? 0;
        $ds = $hoy->TOTAL_DESCUENTOS ?? 0;
        $tt = $hoy->TOTAL_REPORTE ?? 0;

        // ✅ Conteo total:
        // - paginator tiene total()
        // - collection tiene count()
        $totalRegistros = method_exists($reportes, 'total') ? $reportes->total() : $reportes->count();

        // ✅ Es vendedor y no hay reporte hoy
        $sinReporteHoy = $rol === 'VENDEDOR' && $totalRegistros === 0;
    @endphp

    <div class="rep-wrap">

        <div class="rep-head">
            <div class="rep-title">
                <div class="rep-badge">Reporte</div>
                <h1>Reportes de Ventas</h1>
                <p>
                    @if ($rol === 'ADMIN')
                        Historial diario por métodos de pago (Tulinda).
                    @else
                        Solo se muestra el reporte del día (si está generado).
                    @endif
                </p>

                @if ($reporteHoy)
                    <div class="status-pill ok">
                        Reporte de hoy registrado
                        <span class="tick">✓</span>
                    </div>
                @else
                    <div class="status-pill">
                        Reporte de hoy aún no generado
                    </div>
                @endif
            </div>

            {{-- ✅ Botón generar reporte: solo si hoy NO existe --}}
            @if (!$reporteHoy)
                <form method="POST" action="{{ route('reporte.hoy') }}"
                    onsubmit="return confirm('¿Seguro que deseas generar el reporte de HOY?\n\n⚠️ Esta acción se ejecuta una sola vez y no podrás volver a generarlo.')">
                    @csrf
                    <button type="submit" class="pill-btn primary">
                        Generar reporte de hoy
                    </button>
                </form>
            @else
                <button class="rep-btn disabled" type="button" disabled>
                    Reporte ya generado
                </button>
            @endif
        </div>

        {{-- ✅ ALERTA SI ES VENDEDOR Y NO HAY REPORTE --}}
        @if ($sinReporteHoy)
            <div class="rep-alert warn">
                <b>Reporte pendiente.</b>
                <div class="mini">Aún no se ha generado el reporte de hoy. Cuando se genere, aparecerá aquí.</div>
            </div>
        @endif

        {{-- KPIs (si no hay reporte, salen en 0) --}}
        <div class="kpi-grid">
            <div class="kpi-card">
                <div class="kpi-label">
                    @if ($rol === 'ADMIN')
                        Total (último reporte)
                    @else
                        Total del día
                    @endif
                </div>
                <div class="kpi-value">S/ {{ number_format($tt, 2) }}</div>
                <div class="kpi-sub">Ventas totales (incluye todo)</div>
            </div>

            <div class="kpi-card">
                <div class="kpi-label">Efectivo</div>
                <div class="kpi-value">S/ {{ number_format($ef, 2) }}</div>
                <div class="kpi-sub">Caja (cash)</div>
            </div>

            <div class="kpi-card">
                <div class="kpi-label">Yape / Plin</div>
                <div class="kpi-value">S/ {{ number_format($yp, 2) }}</div>
                <div class="kpi-sub">Transferencias</div>
            </div>

            <div class="kpi-card">
                <div class="kpi-label">Tarjeta</div>
                <div class="kpi-value">S/ {{ number_format($tj, 2) }}</div>
                <div class="kpi-sub">POS</div>
            </div>

            <div class="kpi-card warn">
                <div class="kpi-label">Descuentos</div>
                <div class="kpi-value">S/ {{ number_format($ds, 2) }}</div>
                <div class="kpi-sub">Total descuentos</div>
            </div>
        </div>

        {{-- Tabla --}}
        <div class="rep-card">
            <div class="rep-card-head">
                <div>
                    <h2>
                        @if ($rol === 'ADMIN')
                            Historial
                        @else
                            Reporte de hoy
                        @endif
                    </h2>
                    <span class="chip">{{ $totalRegistros }} registros</span>
                </div>

                <div class="muted">
                    @if ($rol === 'ADMIN')
                        Ordenado del más reciente al más antiguo
                    @else
                        Solo el reporte del día
                    @endif
                </div>
            </div>

            <div class="table-wrap">
                <table class="rep-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Fecha</th>
                            <th class="num">Efectivo</th>
                            <th class="num">Yape/Plin</th>
                            <th class="num">Tarjeta</th>
                            <th class="num">Descuentos</th>
                            <th class="num total">Total</th>
                            <th>Empleado</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($reportes as $r)
                            <tr>
                                <td>#{{ $r->ID_REPORTE }}</td>
                                <td>{{ \Carbon\Carbon::parse($r->FECHA_REPORTE)->format('d/m/Y H:i') }}</td>
                                <td class="num">S/ {{ number_format($r->EFECTIVO_REPORTE, 2) }}</td>
                                <td class="num">S/ {{ number_format($r->YAPE_PLIN_REPORTE, 2) }}</td>
                                <td class="num">S/ {{ number_format($r->TARJETA_REPORTE, 2) }}</td>
                                <td class="num">S/ {{ number_format($r->TOTAL_DESCUENTOS, 2) }}</td>
                                <td class="num total">S/ {{ number_format($r->TOTAL_REPORTE, 2) }}</td>
                                <td class="emp">{{ $r->ID_EMPLEADO }}</td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="8" class="empty">
                                    @if ($rol === 'VENDEDOR')
                                        No hay reporte de hoy aún.
                                    @else
                                        No hay reportes aún.
                                    @endif
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>

            {{-- ✅ Paginación SOLO ADMIN --}}
            @if ($rol === 'ADMIN' && method_exists($reportes, 'links'))
                <div class="rep-pagination">
                    {{ $reportes->links() }}
                </div>
            @endif
        </div>

    </div>
@endsection
