@extends('layouts.app')
@section('title', 'Nueva Venta')

@push('styles')
    @vite(['resources/css/venta-create.css'])
@endpush

@section('content')
    <div class="vc-page">
        <div class="vc-card">

            {{-- HEADER --}}
            <div class="vc-head">
                <div class="vc-ico">🧾</div>
                <div class="vc-title">
                    <h1>Nueva Venta</h1>
                    <p>Elige categoría → busca producto → registra venta rápida.</p>
                </div>
                <span class="vc-tag">Ventas</span>
            </div>

            <form method="POST" action="{{ route('ventas.store') }}" class="vc-form" autocomplete="off">
                @csrf

                {{-- CATEGORÍA --}}
                <div class="row">
                    <label>Categoría</label>
                    <select id="categoria" class="cat-select" required>
                        <option value="" selected disabled>Seleccionar categoría...</option>
                    </select>
                    <div class="hint2">Tip: primero filtra por categoría para no perderte.</div>
                </div>

                {{-- PRODUCTO (BUSCADOR POR CATEGORÍA) --}}
                <div class="row">
                    <label>Producto</label>

                    <div class="pbox">
                        <input type="text" id="productoSearch" class="psearch"
                            placeholder="Escribe para buscar dentro de la categoría..." autocomplete="off" disabled />
                        <div class="plist" id="productoList" aria-hidden="true"></div>
                    </div>

                    {{-- SELECT REAL para el POST --}}
                    <select name="ID_PRODUCTO" id="producto" required class="hidden-select">
                        <option value="" disabled selected>Seleccionar...</option>
                        @foreach ($productos as $p)
                            <option value="{{ $p->ID_PRODUCTO }}" data-precio="{{ $p->PRECIO_PRODUCTO }}"
                                data-nombre="{{ $p->TIPO_PRODUCTO }}">
                                {{ $p->ID_PRODUCTO }} — {{ $p->TIPO_PRODUCTO }} — S/
                                {{ number_format($p->PRECIO_PRODUCTO, 2) }}
                            </option>
                        @endforeach
                    </select>

                    <div class="hint3" id="hint-prod">Selecciona una categoría primero.</div>
                </div>

                {{-- CANTIDAD --}}
                <div class="row">
                    <label>Cantidad</label>
                    <input type="number" id="cantidad" name="CANTIDAD" min="1" value="{{ old('CANTIDAD', 1) }}"
                        required data-clear-on-focus="true">
                </div>

                {{-- DESCUENTO --}}
                <div class="row">
                    <label>Descuento (opcional)</label>
                    <input type="number" step="0.01" id="descuento" name="DESCUENTO_PRODUCTO"
                        value="{{ old('DESCUENTO_PRODUCTO', 0) }}" min="0" data-clear-on-focus="true">
                    <div class="hint" id="hint-desc">El descuento no puede superar el subtotal.</div>
                </div>

                {{-- METODO --}}
                <div class="row">
                    <label>Método de pago</label>
                    <select name="METODO_PAGO" id="metodo" required>
                        <option value="EFECTIVO">EFECTIVO</option>
                        <option value="YAPE_PLIN">YAPE/PLIN</option>
                        <option value="TARJETA">TARJETA</option>
                    </select>
                </div>

                {{-- RESUMEN --}}
                <div class="resume">
                    <div class="rline">
                        <span>Precio</span>
                        <b id="r-precio">S/ 0.00</b>
                    </div>
                    <div class="rline">
                        <span>Subtotal</span>
                        <b id="r-subtotal">S/ 0.00</b>
                    </div>
                    <div class="rline">
                        <span>Descuento</span>
                        <b id="r-desc">S/ 0.00</b>
                    </div>
                    <div class="rline total">
                        <span>Total</span>
                        <b id="r-total">S/ 0.00</b>
                    </div>
                </div>

                <div class="actions">
                    <button class="btn-pill primary" type="submit">Guardar</button>
                    <a class="btn-pill ghost" href="{{ route('ventas.index') }}">Volver</a>
                </div>
            </form>
        </div>
    </div>

    <script>
        (function() {
            const selectReal = document.getElementById('producto');
            const catSelect = document.getElementById('categoria');

            const search = document.getElementById('productoSearch');
            const list = document.getElementById('productoList');
            const hintProd = document.getElementById('hint-prod');

            const cantidad = document.getElementById('cantidad');
            const descuento = document.getElementById('descuento');

            const rPrecio = document.getElementById('r-precio');
            const rSub = document.getElementById('r-subtotal');
            const rDesc = document.getElementById('r-desc');
            const rTotal = document.getElementById('r-total');
            const hintDesc = document.getElementById('hint-desc');

            const money = (n) => 'S/ ' + (Number(n || 0).toFixed(2));

            // ===== Obtener items desde el select real =====
            const allItems = Array.from(selectReal.options)
                .filter(o => o.value)
                .map(o => ({
                    id: o.value,
                    categoria: (o.dataset.nombre || '').trim(), // usa TIPO_PRODUCTO como categoría
                    precio: Number(o.dataset.precio || 0),
                    label: o.textContent.trim()
                }));

            // ===== Armar categorías únicas =====
            const cats = Array.from(new Set(allItems.map(i => i.categoria))).sort((a, b) => a.localeCompare(b));
            cats.forEach(c => {
                const opt = document.createElement('option');
                opt.value = c;
                opt.textContent = c;
                catSelect.appendChild(opt);
            });

            let activeIndex = -1;
            let currentShown = [];
            let currentCategory = '';

            // ✅ NUEVO: recordar texto seleccionado para limpiarlo al tocar el input
            let lastSelectedText = "";

            function getPrecio() {
                const opt = selectReal.options[selectReal.selectedIndex];
                return opt ? Number(opt.dataset.precio || 0) : 0;
            }

            function calc() {
                const precio = getPrecio();
                const cant = Math.max(1, Number(cantidad.value || 1));
                let desc = Math.max(0, Number(descuento.value || 0));
                const subtotal = precio * cant;

                if (desc > subtotal) {
                    desc = subtotal;
                    descuento.value = subtotal.toFixed(2);
                    hintDesc.style.display = 'block';
                } else {
                    hintDesc.style.display = 'none';
                }

                const total = Math.max(0, subtotal - desc);

                rPrecio.textContent = money(precio);
                rSub.textContent = money(subtotal);
                rDesc.textContent = money(desc);
                rTotal.textContent = money(total);
            }

            // ===== UI LIST =====
            function openList() {
                list.classList.add('open');
                list.setAttribute('aria-hidden', 'false');
            }

            function closeList() {
                list.classList.remove('open');
                list.setAttribute('aria-hidden', 'true');
                activeIndex = -1;
            }

            function render(results) {
                currentShown = results;
                activeIndex = -1;

                if (!results.length) {
                    list.innerHTML = `<div class="pempty">No hay resultados en esta categoría.</div>`;
                    openList();
                    return;
                }

                list.innerHTML = results.map((it, idx) => `
                    <button type="button" class="pitem" data-idx="${idx}">
                        <div class="p-left">
                            <div class="p-id">${it.id}</div>
                            <div class="p-type">${it.categoria}</div>
                        </div>
                        <div class="p-price">S/ ${it.precio.toFixed(2)}</div>
                    </button>
                `).join('');

                openList();
            }

            function filter(q) {
                const term = (q || '').trim().toLowerCase();
                const pool = allItems.filter(i => i.categoria === currentCategory);

                if (!term) return pool.slice(0, 12);

                const res = pool.filter(it => {
                    const priceText = it.precio.toFixed(2);
                    return (
                        it.id.toLowerCase().includes(term) ||
                        it.categoria.toLowerCase().includes(term) ||
                        priceText.includes(term)
                    );
                });

                return res.slice(0, 12);
            }

            function selectItem(it) {
                selectReal.value = it.id;

                // ✅ guarda el texto seleccionado
                lastSelectedText = `${it.id} — ${it.categoria} — S/ ${it.precio.toFixed(2)}`;
                search.value = lastSelectedText;

                closeList();
                calc();

                // ✅ PRO: pasar directo a cantidad para escribir rápido
                setTimeout(() => cantidad.focus(), 0);
            }

            // click item
            list.addEventListener('click', (e) => {
                const btn = e.target.closest('.pitem');
                if (!btn) return;
                const idx = Number(btn.dataset.idx);
                const it = currentShown[idx];
                if (it) selectItem(it);
            });

            // ✅ NUEVO: al tocar el input, si ya había producto elegido => limpiar para buscar otro
            search.addEventListener('pointerdown', () => {
                if (search.disabled) return;

                if (search.value && search.value === lastSelectedText) {
                    search.value = "";
                    selectReal.value = "";
                    lastSelectedText = "";
                    calc();
                }
            });

            // typing
            search.addEventListener('input', () => render(filter(search.value)));

            // focus open
            search.addEventListener('focus', () => render(filter(search.value)));

            // keyboard
            search.addEventListener('keydown', (e) => {
                if (!list.classList.contains('open')) return;

                const btns = Array.from(list.querySelectorAll('.pitem'));
                if (!btns.length) return;

                if (e.key === 'ArrowDown') {
                    e.preventDefault();
                    activeIndex = Math.min(activeIndex + 1, btns.length - 1);
                    btns.forEach(b => b.classList.remove('active'));
                    btns[activeIndex].classList.add('active');
                    btns[activeIndex].scrollIntoView({
                        block: 'nearest'
                    });
                }

                if (e.key === 'ArrowUp') {
                    e.preventDefault();
                    activeIndex = Math.max(activeIndex - 1, 0);
                    btns.forEach(b => b.classList.remove('active'));
                    btns[activeIndex].classList.add('active');
                    btns[activeIndex].scrollIntoView({
                        block: 'nearest'
                    });
                }

                if (e.key === 'Enter') {
                    e.preventDefault();
                    if (activeIndex >= 0) {
                        const it = currentShown[activeIndex];
                        if (it) selectItem(it);
                    }
                }

                if (e.key === 'Escape') closeList();
            });

            // click outside close
            document.addEventListener('click', (e) => {
                if (!e.target.closest('.pbox')) closeList();
            });

            // change category
            catSelect.addEventListener('change', () => {
                currentCategory = catSelect.value;

                // limpiar selección actual
                selectReal.value = "";
                search.value = "";
                lastSelectedText = "";
                hintProd.style.display = "none";

                // habilitar buscador
                search.disabled = false;
                search.placeholder = `Buscar dentro de ${currentCategory}... (Ej: AB, 25)`;

                render(filter(''));
                calc();

                // ✅ foco al buscador
                setTimeout(() => search.focus(), 0);
            });

            // limpiar inputs al focus
            document.querySelectorAll('[data-clear-on-focus="true"]').forEach(el => {
                el.addEventListener('focus', () => {
                    if (el.value === '0' || el.value === '1') el.value = '';
                });
                el.addEventListener('blur', () => {
                    if (el.id === 'cantidad' && el.value === '') el.value = 1;
                    if (el.id === 'descuento' && el.value === '') el.value = 0;
                    calc();
                });
            });

            cantidad.addEventListener('input', calc);
            descuento.addEventListener('input', calc);

            // init
            hintProd.style.display = "block";
            calc();
        })();
    </script>
@endsection
